#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include <linux/kernel.h>	/* printk() */
#include <linux/fs.h>		/* everything... */
#include <linux/errno.h>	/* error codes */
#include <linux/types.h>	/* size_t */
#include <linux/vmalloc.h>
#include <linux/version.h>
#include <linux/debugfs.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <linux/firmware.h>

#include <linux/of_device.h>
#include <linux/of_address.h>
#include <linux/resource.h>

#include <elpspaccdrv.h>
#include "crypto-ram.h"

MODULE_LICENSE("GPL");		// for use debugfs/sysfs for testing

static   u64 kapi_test_src_pa;
static   u32 kapi_test_sz;
static   u64 kapi_test_dst_pa;

static ST_CYPR_KEY lkey;
static ST_CYPR_PRM lprm;
static ST_CYPR_TRN ltrn;

static char *src_va, *dst_va;

typedef struct _pattern {
	u32 key_size; //128 or 256bits
	bool enc;     // enc=0 for decrypt, enc=1 for encrypt
	u32 unit_size;
	u32 total_size;
	char key[64];
	char tweak[16];
	char ciphertext[96];

} pattern_t;

pattern_t patn_uid1 = {
	.key_size = E_CYPR_KEYL_128,
	.enc = 1,
	.unit_size = 32,
	.total_size = 32,
	.key = {
		0x11,0x22,0x56,0x78,0x12,0x34,0x56,0x78,0x11,0x11,0x11,0x11,0x00,0x00,0x00,0x00,
		0x12,0x34,0x56,0x78,0x12,0x34,0x56,0x78,0x22,0x22,0x22,0x22,0x12,0x34,0x56,0x78},
	.tweak = {
		0x77,0x88,0x01,0x20,0x03,0x40,0x05,0x60,0x07,0x80,0x09,0xa0,0x0b,0xc0,0x0d,0xe0},
	.ciphertext = {}
};

pattern_t patn_uid2 = {
	.key_size = E_CYPR_KEYL_128,
	.enc = 1,
	.unit_size = 32,
	.total_size = 32,
	.key = {
		0x12,0x34,0x56,0x78,0x12,0x34,0x56,0x78,0x11,0x11,0x11,0x11,0x00,0x00,0x00,0x00,
		0x12,0x34,0x56,0x78,0x12,0x34,0x56,0x78,0x22,0x22,0x22,0x22,0x12,0x34,0x56,0x78},
	.tweak = {
		0x34,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00}, //0x1234
	.ciphertext = { //plaintext seq# 00~31
		0xca,0x30,0x00,0x6b,0xd7,0x1a,0xac,0xa2,0x07,0x17,0x5d,0xea,0x5c,0xf9,0x05,0x84,
		0x0c,0x6d,0x04,0x68,0x18,0xcf,0x26,0x33,0x4b,0xaf,0xbd,0xd1,0xf6,0x98,0x8c,0x1a}
};

pattern_t patn_uid3 = {
	.key_size = E_CYPR_KEYL_256,
	.enc = 0,
	.unit_size = 32,
	.total_size = 96,
	.key = {
		0x00,0x00,0x00,0x00,0x00,0x00,0x12,0x34,0x00,0x00,0x00,0x00,0x00,0x00,0x55,0xAA,
		0x12,0x34,0x56,0x78,0x12,0x34,0x56,0x78,0x11,0x11,0x11,0x11,0x00,0x00,0x00,0x00,
		0x88,0x88,0x88,0x88,0x77,0x77,0x77,0x77,0x66,0x66,0x66,0x66,0x55,0x55,0x55,0x55,
		0x44,0x44,0x44,0x44,0x33,0x33,0x33,0x33,0x22,0x22,0x22,0x22,0x11,0x11,0x11,0x11},

	.tweak = {
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x34,0x12,0x00,0x00,0x00,0x00,0x00,0x00
	},
	.ciphertext= { //plaintext seq# 00~95
		0x1a, 0x38, 0xa7, 0x09, 0x22, 0x45, 0x6d, 0x42, 0xc9, 0x03, 0x42, 0x9d, 0x44, 0x62, 0xc0, 0xad,
		0xb6, 0x1e, 0x1c, 0x95, 0xf2, 0x10, 0x09, 0x0f, 0xfe, 0x3d, 0xed, 0x9c, 0xa1, 0x79, 0x07, 0x45,
		0xec, 0x79, 0xee, 0x49, 0xec, 0x77, 0x4c, 0xbd, 0x2b, 0x5e, 0x55, 0xa4, 0x05, 0x7b, 0x4e, 0x25,
		0x4c, 0x3c, 0x73, 0x8b, 0x90, 0x2d, 0xc3, 0x7e, 0x2c, 0xf7, 0x0a, 0x88, 0x97, 0xd7, 0xc7, 0xf1,
		0xc8, 0x09, 0x14, 0xea, 0xef, 0xc4, 0x84, 0x7a, 0xa0, 0x27, 0x6c, 0x23, 0xa3, 0xa7, 0xea, 0x5f,
		0x89, 0x35, 0x2e, 0x88, 0x67, 0x13, 0x50, 0x9e, 0xf9, 0x52, 0x6f, 0xf9, 0x9b, 0xc7, 0x52, 0xa8

	},
};

pattern_t patn_uid4 = { //include 2 pkts: patn_uid4_1, patn_uid4_2
	.key_size = E_CYPR_KEYL_128,
	.enc = 1,
	.unit_size = 32,
	.total_size = 64,
	.key = {
		0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,
		0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10
		},

	.tweak = {
		0x34,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	.ciphertext= { //plaintext seq# 00~31
		0x8a, 0xc6, 0x0a, 0xbf, 0xfe, 0x9e, 0x0c, 0xb8, 0x72, 0x7f, 0x07, 0x76, 0xba, 0x59, 0x47, 0x73,
		0x11, 0x67, 0x38, 0xaa, 0x80, 0x36, 0x6e, 0xa5, 0x9c, 0x2a, 0x98, 0x70, 0xf6, 0xdf, 0x73, 0xcf,
	               //plaintext seq# 32~63
		0xa5, 0x50, 0x1e, 0xf3, 0x56, 0xc2, 0x31, 0xd0, 0xe8, 0x8e, 0x82, 0x75, 0xb2, 0x42, 0x70, 0x72,
		0xe4, 0xcf, 0x68, 0xb6, 0x00, 0xcd, 0x48, 0x7b, 0x93, 0xa6, 0x1d, 0x11, 0x6c, 0x09, 0x0d, 0x72
	}
};

pattern_t patn_uid4_1 = { //1st pkt, used in uid4 for encrypt test and uid5 for decrypt test
	.key_size = E_CYPR_KEYL_128,
	.enc = 1,
	.unit_size = 32,
	.total_size = 32,
	.key = {
		0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,
		0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10
		},

	.tweak = {
		0x34,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	.ciphertext= { //plaintext seq# 00~31
		0x8a, 0xc6, 0x0a, 0xbf, 0xfe, 0x9e, 0x0c, 0xb8, 0x72, 0x7f, 0x07, 0x76, 0xba, 0x59, 0x47, 0x73,
		0x11, 0x67, 0x38, 0xaa, 0x80, 0x36, 0x6e, 0xa5, 0x9c, 0x2a, 0x98, 0x70, 0xf6, 0xdf, 0x73, 0xcf
	}
};

pattern_t patn_uid4_2 = { //2nd pkt, used in uid4 for encrypt test and uid5 for decrypt test
	.key_size = E_CYPR_KEYL_128,
	.enc = 1,
	.unit_size = 32,
	.total_size = 32,
	.key = {
		0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,
		0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10,0xfe,0xdc,0xba,0x98,0x76,0x54,0x32,0x10
		},

	.tweak = {
		0x35,0x12,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	.ciphertext= { //plaintext seq# 32~63
		0xa5, 0x50, 0x1e, 0xf3, 0x56, 0xc2, 0x31, 0xd0, 0xe8, 0x8e, 0x82, 0x75, 0xb2, 0x42, 0x70, 0x72,
		0xe4, 0xcf, 0x68, 0xb6, 0x00, 0xcd, 0x48, 0x7b, 0x93, 0xa6, 0x1d, 0x11, 0x6c, 0x09, 0x0d, 0x72
	}
};

pattern_t patn_uid6 = {
	.key_size = E_CYPR_KEYL_128,
	.enc = 1, //encrypt, then decrypt
	.unit_size = 256,
	.total_size = 10*1024*1024, //10MB
	.key = {
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f
	},

	.tweak = {
		0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0x34,0x12,0x00,0x00,0x00,0x00,0x00,0x00
	},
	.ciphertext= { //plaintext seq# 00~255, repeat, and up to 10MB
		       //do not embedded 10MB ciphertext
	},
};


static void init_seq_src(char *addr, int size) //sequence number 0~255
{
	int i;
	for(i=0;i<size;i++)
		*addr++ = (i&0xFF);
}

static int cmp_seq_dst(char *addr, int size) //sequence number 0~255
{
	int i;
	for(i=0;i<size;i++)
		if((*addr++)!= (i&0xFF))
			return -1;
	return 0;
}


static void set_param_src(pattern_t *p)
{
	lkey.e_keyLType = p->key_size;
	if(p->key_size == E_CYPR_KEYL_128)
	{
		memcpy(&lkey.ui32_Key1_0, p->key, 16); //128bit = 16B, AES-XTS need 2 keys
		memcpy(&lkey.ui32_Key2_0, p->key+16, 16); //128bit = 16B, AES-XTS need 2 keys
	}
	else
	{
		memcpy(&lkey.ui32_Key1_0, p->key, 32); //256bit = 32B, AES-XTS need 2 keys
		memcpy(&lkey.ui32_Key2_0, p->key+32, 32); //256bit = 32B, AES-XTS need 2 keys
	}

	lprm.e_operation = p->enc?E_CYPR_OPR_ENC:E_CYPR_OPR_DEC;

	lprm.ui32_unitSize = p->unit_size;

	memcpy(&lprm.ui32_unitNum_0, p->tweak, 16); //tweak need 16B
	lprm.ui32_processSize = p->total_size;

	ltrn.ui32_srcAdrs_0 = (u32)kapi_test_src_pa;
	ltrn.ui32_dstAdrs_0 = (u32)kapi_test_dst_pa;
	ltrn.ui32_srcAdrs_1 = (u32)(kapi_test_src_pa >> 32);
	ltrn.ui32_dstAdrs_1 = (u32)(kapi_test_dst_pa >> 32);

	if(p->enc)
		init_seq_src(src_va, p->total_size);
	else
		memcpy((void *)(uintptr_t)src_va, p->ciphertext, p->total_size);

}

static int test_uid1(void)
{
	int err;
	pattern_t *p =  &patn_uid1;

	set_param_src(p);

	err = cryptRam_init();
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);


	lkey.e_keyLType = E_CYPR_KEYL_128|0x100; //invalid key length
	err = cryptRam_setKey(&lkey);
	if (err != E_CYPR_ERR_KEYL)
	{
		printk("%s[L%d]: E_CYPR_ERR_KEYL failed!\n", __func__, __LINE__);
		return -1;
	}

	lkey.e_keyLType = E_CYPR_KEYL_128; //revert

	lprm.ui32_processSize = p->total_size + 1; //invalid proc size
	err = cryptRam_start(&lprm, &ltrn);
	if (err != E_CYPR_ERR_PROC_LEN)
	{
		printk("%s[L%d]: E_CYPR_ERR_PROC_LEN failed!\n", __func__, __LINE__);
		return -1;
	}

	lprm.ui32_processSize = p->total_size; //revert

	lprm.e_operation = E_CYPR_OPR_ENC|0xF; //invalid operation mode
	err = cryptRam_start(&lprm, &ltrn);
	if (err != E_CYPR_ERR_PARAM)
	{
		printk("%s[L%d]: E_CYPR_ERR_PARAM failed!\n", __func__, __LINE__);
		return -1;
	}


	printk("%s[L%d]: Pass!\n", __func__, __LINE__);

	return 0;
}

static int test_uid2(void)
{
	int err;
	pattern_t *p =  &patn_uid2;

	set_param_src(p);

	err = cryptRam_init();
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

	err = cryptRam_setKey(&lkey);
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

	err = cryptRam_start(&lprm, &ltrn);
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);
		return -1;
	}

	if(memcmp((const void *)(uintptr_t)p->ciphertext, dst_va, lprm.ui32_processSize))
	{//failed, then dump all output memory
		print_hex_dump(KERN_INFO, "Input: ", DUMP_PREFIX_ADDRESS, 16, 1,src_va,lprm.ui32_processSize , false);
		print_hex_dump(KERN_INFO, "HW result: ", DUMP_PREFIX_ADDRESS, 16, 1,dst_va,lprm.ui32_processSize , false);
		return -1;
	}
	else //pass
	{
		printk("%s[L%d]: Pass!\n", __func__, __LINE__);
	}

	return 0;
}

static int test_uid3(void)
{
	int err;
	pattern_t *p =  &patn_uid3;

	set_param_src(p);

	err = cryptRam_init();
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

	err = cryptRam_setKey(&lkey);
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

	err = cryptRam_start(&lprm, &ltrn);
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);
		return -1;
	}

	if(cmp_seq_dst(dst_va, lprm.ui32_processSize))
	{//failed, then dump all output memory
		print_hex_dump(KERN_INFO, "HW result: ", DUMP_PREFIX_ADDRESS, 16, 1,dst_va,lprm.ui32_processSize , false);
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);
		return -1;
	}
	else //pass
	{
		printk("%s[L%d]: Pass!\n", __func__, __LINE__);
	}

	return 0;
}

static int test_uid4(void)
{
	int err;
	pattern_t *p =  &patn_uid4;

	set_param_src(p);

	err = cryptRam_init();
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);

	err = cryptRam_setKey(&lkey);
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);

////////////////////////////////////////////////////////////////////////////////
	p =  &patn_uid4_1;
//Adjust processSize to process 1st pkt only
	lprm.ui32_processSize = p->total_size;
////////////////////////////////////////////////////////////////////////////////

	err = cryptRam_start(&lprm, &ltrn); //1st cryptRam_start()
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);
		return -1;
	}
////////////////////////////////////////////////////////////////////////////////
	p =  &patn_uid4_2;
//Adjust tweak, src, dst, processSize to process 2nd pkt only
	memcpy(&lprm.ui32_unitNum_0, p->tweak, 16); //tweak need 16B
	lprm.ui32_processSize = p->total_size;

	ltrn.ui32_srcAdrs_0 = kapi_test_src_pa + p->unit_size;
	ltrn.ui32_dstAdrs_0 = kapi_test_dst_pa + p->unit_size;
////////////////////////////////////////////////////////////////////////////////

	// test key reuse
	err = cryptRam_start(&lprm, &ltrn); //2nd cryptRam_start()
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);
		return -1;
	}


	p = &patn_uid4;
	if(memcmp((const void *)(uintptr_t)p->ciphertext, dst_va, p->total_size))
	{//failed, then dump all output memory
		print_hex_dump(KERN_INFO, "HW result: ", DUMP_PREFIX_ADDRESS, 16, 1,dst_va,p->total_size, false);
		return -1;
	}
	else //pass
	{
		printk("%s[L%d]: Pass!\n", __func__, __LINE__);
	}


	return 0;
}

static int test_uid5(void)
{
	int err;
	pattern_t *p =  &patn_uid4;

	p->enc = 0; //use patn_uid4 for decode test

	set_param_src(p);

	err = cryptRam_init();
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);

	err = cryptRam_setKey(&lkey);
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

////////////////////////////////////////////////////////////////////////////////
	p =  &patn_uid4_1;
//Adjust processSize to process 1st pkt only
	lprm.ui32_processSize = p->total_size;
////////////////////////////////////////////////////////////////////////////////

	err = cryptRam_start(&lprm, &ltrn); //1st cryptRam_start()
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);
		return -1;
	}
////////////////////////////////////////////////////////////////////////////////
	p =  &patn_uid4_2;
//Adjust tweak, src, dst to process 2nd pkt only
	memcpy(&lprm.ui32_unitNum_0, p->tweak, 16); //tweak need 16B

	ltrn.ui32_srcAdrs_0 = kapi_test_src_pa + p->unit_size;
	ltrn.ui32_dstAdrs_0 = kapi_test_dst_pa + p->unit_size;
////////////////////////////////////////////////////////////////////////////////

	// test key reuse
	err = cryptRam_start(&lprm, &ltrn); //2nd cryptRam_start()
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);
		return -1;
	}

	p = &patn_uid4;

	if(cmp_seq_dst(dst_va, p->total_size))
	{//failed, then dump all output memory
		print_hex_dump(KERN_INFO, "HW result: ", DUMP_PREFIX_ADDRESS, 16, 1,dst_va,p->total_size, false);
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);
		return -1;
	}
	else //pass
	{
		printk("%s[L%d]: Pass!\n", __func__, __LINE__);
	}

	return 0;
}

static int test_uid6(void)
{
	int err;
	pattern_t *p =  &patn_uid6;
	ktime_t calltime, delta, rettime;
	ktime_t starttime;
	unsigned long duration_enc, duration_dec;


	set_param_src(p);

	calltime = ktime_get();
	err = cryptRam_init();
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

	err = cryptRam_setKey(&lkey);
	if (err != E_CYPR_ERR_OK)
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);

	starttime = ktime_get();
	err = cryptRam_start(&lprm, &ltrn); //encrypt
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);
		return -1;
	}

	rettime = ktime_get();
	delta = ktime_sub(rettime, calltime);
	duration_enc = ktime_to_ns(delta)/1000; //convert ns to us

	printk("calltime=%lld, starttime=%lld, rettime=%lld\n", calltime.tv64, starttime.tv64, rettime.tv64);

#ifdef PERF_MON
	if(spacc_kt[0][0])
	{
		int i;
		for(i=0; i< kt_idx; i++)
		{
			printk("kt0=%lld, kt1=%lld, kt2=%lld, kt3=%lld, kt4=%lld, kt5=%lld, kt6=%lld, kt7=%lld, irq_start=%lld, irq_end=%lld, kt8=%lld, kt9=%lld, kt10=%lld, total=%lld\n",
			       spacc_kt[i][0],
			       (spacc_kt[i][1] -  spacc_kt[i][0])/1000,
			       (spacc_kt[i][2] -  spacc_kt[i][1])/1000,
			       (spacc_kt[i][3] -  spacc_kt[i][2])/1000,
			       (spacc_kt[i][4] -  spacc_kt[i][3])/1000,
			       (spacc_kt[i][5] -  spacc_kt[i][4])/1000,
			       (spacc_kt[i][6] -  spacc_kt[i][5])/1000,
			       (spacc_kt[i][7] -  spacc_kt[i][6])/1000,
			       (spacc_kt[i][11] - spacc_kt[i][7])/1000,
			       (spacc_kt[i][12] - spacc_kt[i][7])/1000,
			       (spacc_kt[i][8] -  spacc_kt[i][12])/1000,
			       (spacc_kt[i][9] -  spacc_kt[i][8])/1000,
			       (spacc_kt[i][10] - spacc_kt[i][9])/1000,
			       (spacc_kt[i][10] - spacc_kt[i][0])/1000
			      );
		}
	}
#endif


	calltime = ktime_get();
	lprm.e_operation = E_CYPR_OPR_DEC;
	ltrn.ui32_srcAdrs_0 = kapi_test_dst_pa;
	ltrn.ui32_dstAdrs_0 = kapi_test_dst_pa;

	// key reuse
	starttime = ktime_get();
	err = cryptRam_start(&lprm, &ltrn); //decrypt
	if (err != E_CYPR_ERR_OK)
	{
		printk("%s[L%d]: err! %d\n", __func__, __LINE__, err);
		return -1;
	}
	rettime = ktime_get();
	delta = ktime_sub(rettime, calltime);
	duration_dec = ktime_to_ns(delta)/1000; //convert ns to us

	printk("calltime=%lld, starttime=%lld, rettime=%lld\n", calltime.tv64, starttime.tv64, rettime.tv64);
#ifdef PERF_MON
	if(spacc_kt[0][0])
	{
		int i;
		for(i=0; i< kt_idx; i++)
		{
			printk("kt0=%lld, kt1=%lld, kt2=%lld, kt3=%lld, kt4=%lld, kt5=%lld, kt6=%lld, kt7=%lld, irq_start=%lld, irq_end=%lld, kt8=%lld, kt9=%lld, kt10=%lld, total=%lld\n",
			       spacc_kt[i][0],
			       (spacc_kt[i][1] -  spacc_kt[i][0])/1000,
			       (spacc_kt[i][2] -  spacc_kt[i][1])/1000,
			       (spacc_kt[i][3] -  spacc_kt[i][2])/1000,
			       (spacc_kt[i][4] -  spacc_kt[i][3])/1000,
			       (spacc_kt[i][5] -  spacc_kt[i][4])/1000,
			       (spacc_kt[i][6] -  spacc_kt[i][5])/1000,
			       (spacc_kt[i][7] -  spacc_kt[i][6])/1000,
			       (spacc_kt[i][11] - spacc_kt[i][7])/1000,
			       (spacc_kt[i][12] - spacc_kt[i][7])/1000,
			       (spacc_kt[i][8] -  spacc_kt[i][12])/1000,
			       (spacc_kt[i][9] -  spacc_kt[i][8])/1000,
			       (spacc_kt[i][10] - spacc_kt[i][9])/1000,
			       (spacc_kt[i][10] - spacc_kt[i][0])/1000
			      );
		}
	}
#endif

	if(cmp_seq_dst(dst_va, lprm.ui32_processSize))
	{//failed, then dump all output memory
		print_hex_dump(KERN_INFO, "HW result: ", DUMP_PREFIX_ADDRESS, 16, 1,dst_va,lprm.ui32_processSize , false);
		printk("%s[L%d]: Failed!\n", __func__, __LINE__);
		return -1;
	}
	else //pass
	{
		u32 sz = p->total_size;
		printk
			("%s[L%d]: Pass! %dMB with unit_size=%d encode/decodec took %ld/%ld usec => %lu.%02luMB/s, %lu.%02luMB/s\n",
			 __func__, __LINE__, sz >> 20, p->unit_size, duration_enc,
			 duration_dec, sz / duration_enc,
			 (100 * sz / duration_enc) % 100, sz / duration_dec,
			 (100 * sz / duration_dec) % 100);
	}

	return 0;
}


static int kapitest_remove(struct platform_device *pdev)
{
	return 0;
}

static int kapitest_probe(struct platform_device *pdev)
{
	struct device_node *np;
        const __be32 *prop_val;
        u64 prop_size = 0;

	np = of_find_node_by_path("/reserved-memory/spacc,testbuf");
	if (!np)
	{
		printk("%s[%d]: of_find_node_by_path() got NULL\n", __func__, __LINE__);
		return false;
	}

        prop_val = of_get_address(np, 0, &prop_size, NULL);
        if (!prop_val)
	{
		printk("%s[%d]: of_get_address() got error\n", __func__, __LINE__);
		return -EINVAL;
	}

        /* We assume space for four 32-bit arguments */
        if (prop_size < 4 * sizeof(u32) || prop_size > (u64)ULONG_MAX)
	{
		printk("%s[%d]: prop_size error\n", __func__, __LINE__);
                return -EINVAL;
	}

        kapi_test_src_pa = (u32)be64_to_cpup((const __be64 *)prop_val);
        if (!kapi_test_src_pa)
	{
		printk("%s[%d]: be64_to_cpup() got error\n", __func__, __LINE__);
                return -EINVAL;
	}
	kapi_test_sz = (u32)prop_size;
	// use 64-bit DMA address for ES; ref to kernel> drivers/udif/mach-cxd900xx/adrs_conv.c
        kapi_test_src_pa |= 0x400000000;

	kapi_test_dst_pa = kapi_test_src_pa+kapi_test_sz/2;

	printk("%s[%d]: src_pa(%llx), dst_pa(%llx), sz(%08x)\n", __func__, __LINE__, kapi_test_src_pa, kapi_test_dst_pa, kapi_test_sz);
	src_va =
	    devm_ioremap_nocache(&pdev->dev, kapi_test_src_pa, kapi_test_sz);

	dst_va = src_va + kapi_test_sz/2;
	printk("%s[%d]: src_va(%p), dst_va(%p), sz(%08x)\n", __func__, __LINE__, src_va, dst_va, kapi_test_sz);

	test_uid1();
	test_uid2();
	test_uid3();
	test_uid4();
	test_uid5();
	test_uid6();

	return 0;
}

static void kapitest_release(struct device *dev)
{
}

static struct platform_driver mydev_driver = {
	.driver = {
		   .name = "kapi_test",
		   },
	.probe = kapitest_probe,
	.remove = kapitest_remove,
};

static struct platform_device mydev_pdevice = {
	.name = "kapi_test",
	.dev = {
		.release = kapitest_release,
		}
};

static int __init kapitest_init(void)
{
	if (platform_driver_register(&mydev_driver) == 0)
		(void)platform_device_register(&mydev_pdevice);
	return 0;
}

static void __exit kapitest_exit(void)
{
	platform_device_unregister(&mydev_pdevice);
	platform_driver_unregister(&mydev_driver);
}

module_init(kapitest_init);
module_exit(kapitest_exit);
