/*
 *  err3425265.c
 *
 *    PCIeDMAC ERRATA3425265
 *
 *  Copyright 2021 Sony Corporation.
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 *
 */
#include "pcie_dma.h"

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/kthread.h>
#include <linux/interrupt.h>
#include "regs.h"

#define TSK_NAME "pcidmamon"

#define STALL_WAIT	 100 /* usec */
#define PATROL_WAIT	1000 /* usec */

static uint32_t pcidma_readl(volatile void __iomem *addr)
{
	return pcie_readl(addr);
}

/* Is the state of DMAC suspected of stall ? */
static inline bool is_suspect(struct dma_info_t *chan)
{
	struct dma_reg_t *reg;
	uint32_t ctrl;

	reg = (PCIDMA_RD == chan->stat.dir) ? &chan->dmard : &chan->dmawr;
	/* read the state of DMAC */
	ctrl = pcidma_readl(reg->ctl + DMA_CTRL1);
	return ((ctrl & (DMA_CTRL1_LLE|DMA_CTRL1_CS|DMA_CTRL1_CCS|DMA_CTRL1_CB))
		== (DMA_CTRL1_LLE|DMA_CTRL1_CS_RUN));
}

struct stall_info {
	struct stall_stat {
		int ok;
		/* bitmap of stall suspect channels */
		unsigned int bitmap[2]; /* ST_PHASE1, ST_PHASE2 */
		int cnt; /* number of stall suspect channels */
		unsigned int result;
	} stat[2]; /* ST_RD, ST_WR */
	unsigned long done_cnt[PCIDMA_CH_MAX]; /* count of DMA finish */
};
/* stat[] */
#define ST_RD		0
#define ST_WR		1
/* bitmap[] */
#define ST_PHASE1	0
#define ST_PHASE2	1

static int stall_check(struct pci_info_t *p, int phase, struct stall_info *info)
{
	int ch, i;
	struct dma_info_t *chan;
	struct stall_stat *st;

	info->stat[ST_RD].cnt = info->stat[ST_WR].cnt = 0;
	for (ch=0, chan=p->dma_info; ch < PCIDMA_CH_MAX; ch++, chan++) {
		unsigned long flags;

		LOCK(chan);
		if (pcidma_busy(chan) && chan->stat.ll_mode) { /* check LL-mode DMA */
			st = (PCIDMA_RD == chan->stat.dir) ? &info->stat[ST_RD] : &info->stat[ST_WR];
			if (ST_PHASE2 == phase  &&  !st->ok) {
				if (!(st->bitmap[ST_PHASE1] & (1U << ch))) {
					/* newcomer in phase2. not stall */
					st->ok = 1;
				} else if (info->done_cnt[ch] != chan->stat.done_cnt) {
					/* already finish. not stall */
					st->ok = 1;
				}
			}
			if (!st->ok) {
				if (is_suspect(chan)) {
					/* This channel is a stall suspect. */
					st->bitmap[phase] |= 1U << ch;
					st->cnt++;
					/* save the count of DMA finish. */
					info->done_cnt[ch] = chan->stat.done_cnt;
				} else {
					/* This channel is OK. not stall */
					st->ok = 1;
				}
			}
		}
		UNLOCK(chan);
		if (info->stat[ST_RD].ok && info->stat[ST_WR].ok)
			break;
	}

	for (i = ST_RD; i <= ST_WR; i++) {
		if (info->stat[i].ok)
			continue;
		if (info->stat[i].cnt < 2) {
			/* The stall is caused by two or more LL-DMA. not stall */
			info->stat[i].ok = 1;
			continue;
		}
		if (ST_PHASE2 == phase
		    && info->stat[i].bitmap[ST_PHASE1] != info->stat[i].bitmap[ST_PHASE2]) {
			/* the stall suspects is changed. not stall */
			info->stat[i].ok = 1;
			continue;
		}
		info->stat[i].result = info->stat[i].bitmap[ST_PHASE1];
	}
	return (info->stat[ST_RD].ok && info->stat[ST_WR].ok);
}

static int stallmon(void *data)
{
	int pci, i, changed;
	struct pci_info_t *p;
	struct stall_info info;
	unsigned int prev[PCIDMA_PCIE_MAX][2] = { 0 };
	char buf[64];

	while (!kthread_should_stop()) {
		for (pci=0, p=pcie_dmacs; pci < PCIDMA_PCIE_MAX; pci++, p++) {
			if (!p->dmac || !(p->flag & FLAG_IRQ) || !p->state)
				continue;

			/*--- Find stall suspect channels ---*/
			memset(&info, 0, sizeof info);
			if (!stall_check(p, ST_PHASE1, &info)) {
				udelay(STALL_WAIT);
				stall_check(p, ST_PHASE2, &info);
			}

			changed = 0;
			for (i = ST_RD; i <= ST_WR; i++) {
				if (info.stat[i].result != prev[pci][i]) {
					changed = 1;
					prev[pci][i] = info.stat[i].result;
				}
			}
			if (!changed)
				continue;

			scnprintf(buf, sizeof buf, "ERR:PCI%d:DMAC:STALL=0x%x,0x%x",
				  p->id, info.stat[ST_RD].result,
				  info.stat[ST_WR].result);
			BOOT_TIME_ADD1(buf);
			printk(KERN_ERR "%s\n", buf);
		}

		usleep_range(PATROL_WAIT, PATROL_WAIT+100);
	}
	return 0;
}

/* stall monitor */
static struct task_struct *stallmon_tsk;

int stallmon_init(void)
{
	struct task_struct *tsk;

	tsk = kthread_run(stallmon, NULL, TSK_NAME);
	if (IS_ERR(tsk)) {
		printk(KERN_ERR "ERROR:%s:can not create kthread\n", __func__);
		return UDIF_ERR_NOMEM;
	}
	stallmon_tsk = tsk;
	return UDIF_ERR_OK;
}

void stallmon_exit(void)
{
	if (stallmon_tsk) {
		kthread_stop(stallmon_tsk);
		stallmon_tsk = NULL;
	}
}
