/*
 * PCIe RC driver for Synopsys DesignWare Core
 *
 * Copyright (C) 2015-2016 Synopsys, Inc. (www.synopsys.com)
 *
 * Authors: Joao Pinto <Joao.Pinto@synopsys.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/of_gpio.h>
#include <linux/pci.h>
#include <linux/of_pci.h>
#include <linux/platform_device.h>
#include <linux/resource.h>
#include <linux/signal.h>
#include <linux/types.h>
#include <linux/reset.h>
#include <linux/snsc_boot_time.h>
#ifdef CONFIG_ARCH_CXD900XX
#include <mach/pcie_export.h>
#include <mach/udif/timer.h>
#include <mach/gic_export.h>
#endif /* CONFIG_ARCH_CXD900XX */

#include "pcie-designware.h"

#define PHY_READY_MAX_TRY  200

//#define DO_NOT_HANDOVER

struct dw_plat_pcie {
    struct dw_pcie		*pci;
#ifndef CONFIG_ARCH_CXD900XX_FPGA
    struct reset_control* prst[E_PLAT_RST_TOTAL];
    struct clk *pclk[E_PLAT_CLK_TOTAL];
    u32 lanes;
    u32 firm;
#endif
};

void __weak pcie_dwc_host_notify(int ch, int state){}

#ifndef CONFIG_ARCH_CXD900XX_FPGA
static const char* rst_ctrl_name[E_PLAT_RST_TOTAL] = {"rst_pwrup","rst_button","rst_pe","rst_apb","rst_axi2avb_pci","rst_axi2avb_pci_i0"};
static const char* clk_gate_name[E_PLAT_CLK_TOTAL] = {"apb_pclk", "dbi_aclk", "slv_aclk", "mstr_aclk", "aux_clk", "axi2avb_pcie0", "axi2avb_pcie1"};
static int dw_plat_suspend_sys(struct dw_plat_pcie* dw_plat_pcie);

/*----- PHY firmware download -----*/
static const u16 firm1[] = {
#include "dwc_c10pcie3phy_phy_x1_ns_pcs_raw_sram.h"
};
struct firm_info {
	const u16 *fw;
	int n;
};
static const struct firm_info firms[] = {
	{ firm1, ARRAY_SIZE(firm1) },
};
#define N_FIRM ARRAY_SIZE(firms)

static void dw_plat_use_sram(struct dw_plat_pcie *dw_plat_pcie)
{
	u32 val;

	if (dw_plat_pcie->pci->pp.handover)
		return;
	if (!dw_plat_pcie->firm)
		return;
	//Use external SRAM
	val = dw_pcie_readl_phy(dw_plat_pcie->pci, PHY_SRAM_CTRL);
	val &= ~(PHY_SRAM_BYPASS|PHY_SRAM_EXT_LD_DONE);
	dw_pcie_writel_phy(dw_plat_pcie->pci, PHY_SRAM_CTRL, val);
	wmb();
}

#define SRAM_INIT_MAX_TRY  200

static int dw_plat_fw_download(struct dw_plat_pcie *dw_plat_pcie)
{
	struct firm_info const *fwinf = firms + dw_plat_pcie->firm - 1;
	int i, ret;
	u32 val, offs;

	if (dw_plat_pcie->pci->pp.handover)
		return 0;
	if (!dw_plat_pcie->firm)
		return 0;
	//Wait for external SRAM ready
	ret = -1;
	i = SRAM_INIT_MAX_TRY;
	while (--i >= 0) {
		udelay(1);
		val = dw_pcie_readl_phy(dw_plat_pcie->pci, PHY_SRAM_CTRL);
		if (val & PHY_SRAM_INIT_DONE)
			break;
	}
	udelay(1); //delay before SRAM access
	if (i >= 0) {
		ret = 0;
		// Update SRAM
		offs = PHY_SRAM_BASE_OFF;
		for (i = 0; i < fwinf->n; i++) {
			val = fwinf->fw[i];
			dw_pcie_writel_phy(dw_plat_pcie->pci, offs, val);
			offs += 4;
		}
		wmb();
		udelay(1); //delay after SRAM access
	}
	val = dw_pcie_readl_phy(dw_plat_pcie->pci, PHY_SRAM_CTRL);
	val |= PHY_SRAM_EXT_LD_DONE;
	dw_pcie_writel_phy(dw_plat_pcie->pci, PHY_SRAM_CTRL, val);
	wmb();
	return ret;
}
/*---------------------------------*/

int __weak pcie_dwc_vboost(int ch)
{
    return -1;
}

int __weak pcie_dwc_tmo(int ch)
{
	return -1;
}

int __weak pcie_dwc_axi_tmo(int ch)
{
	return -1;
}

int __weak pcie_has_prsnt(int ch)
{
	return 0;
}

int __weak pcie_prsnt(int ch)
{
	return 1;
}

int __weak pcie_prsnt_irq(int ch)
{
	return -1;
}

int __weak pcie_prsnt_tmo(int ch)
{
	return 0;
}

int __weak pcie_prsnt_low1(int ch)
{
	return 0;
}

int __weak pcie_prsnt_low2(int ch)
{
	return 0;
}

int __weak pcie_prsnt_high(int ch)
{
	return 0;
}

void __weak pcie_perst_delay(int ch) {}

int __weak pcie_get_perst(int ch)
{
	return 0;
}

static bool dw_plat_is_initialized(struct dw_plat_pcie *p)
{
	return !reset_control_status(p->prst[E_PLAT_PCIE_PWRUP_RST]);
}

static int dw_plat_setup_sys(struct dw_plat_pcie* dw_plat_pcie)
{
    int ret = 0, max_try = 0;
    u32 val;
    int ch = of_get_pci_domain_nr(dw_plat_pcie->pci->dev->of_node);
    enum dw_pcie_phy_status status = DW_PCIE_PHY_IDLE;
    unsigned long long t1;

    dw_plat_pcie->pci->pp.handover = 0;
    /* already initialized ? */
    if (dw_plat_is_initialized(dw_plat_pcie)) {
#ifdef DO_NOT_HANDOVER
	    /* reset */
	reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_PWRUP_RST]);
	reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_BUTTON_RST]);
	reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_PE_RST]);
	reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_APB_RST]);
	udelay(10);
#else
	dw_plat_pcie->pci->pp.handover = 1;
	printk(KERN_ERR "PCI%d: handover\n", ch);
#endif
    }

    //assert APB reset
    //  for plug-in again case
    if (!dw_plat_pcie->pci->pp.handover) {
    ret |= reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_APB_RST]);
    }

    //clock enable
    /* When handover is true, clocks are already enabled.
       But we should call clk_prepare_enable to adjust the reference count. */
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_PCIE_APB_PCLK]);
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_PCIE_DBI_ACLK]);
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_PCIE_SLV_ACLK]);
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_PCIE_MSTR_ACK]);
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_PCIE_AUX_CLK]);

    //wait 10us idle for 1Mhz aux_clk released form clock gated cell
    udelay(10);

    //APB reset release
    ret |= reset_control_deassert(dw_plat_pcie->prst[E_PLAT_PCIE_APB_RST]);

    //PHY use SRAM
    dw_plat_use_sram(dw_plat_pcie);

    //reset release
    ret |= reset_control_deassert(dw_plat_pcie->prst[E_PLAT_PCIE_PWRUP_RST]);
    ret |= reset_control_deassert(dw_plat_pcie->prst[E_PLAT_PCIE_BUTTON_RST]);
    ret |= reset_control_deassert(dw_plat_pcie->prst[E_PLAT_PCIE_PE_RST]);

    wmb();

    //PHY firmware download
    if (dw_plat_fw_download(dw_plat_pcie) < 0) {
	    printk(KERN_ERR "PCI%d: PHY SRAM timeout\n", ch);
    }

    //set IDDQ=0
    val = dw_pcie_readl_phy(dw_plat_pcie->pci, PHY_CONFIG_0X18_OFF);
    if (val & PHY_POWERDOWN) {
	    val &= ~PHY_POWERDOWN;
	    dw_pcie_writel_phy(dw_plat_pcie->pci, PHY_CONFIG_0X18_OFF, val);
	    wmb();
	    udelay(10);
    }

    //CLKREQ# OE
    val = dw_pcie_readl_phy(dw_plat_pcie->pci, GENERAL_SIGNAL_CTRL_OFF);
    if (val & CLKREQ_FW) {
	    val &= ~CLKREQ_FW;
	    dw_pcie_writel_phy(dw_plat_pcie->pci, GENERAL_SIGNAL_CTRL_OFF, val);
    }

    { //set TX_VBOOST_LVL
        int lvl;

        lvl = pcie_dwc_vboost(ch);
        if (0 <= lvl && lvl < 8) {
            val = dw_pcie_readl_phy(dw_plat_pcie->pci, SUP_DIG_LVL_OVRD_IN);
            val &= ~(0xfU << 6);
            val |=  (1U << 9)|(lvl << 6);
            dw_pcie_writel_phy(dw_plat_pcie->pci, SUP_DIG_LVL_OVRD_IN, val);
            printk(KERN_ERR "PCI%d: SUP_DIG_LVL_OVRD_IN=0x%x\n", ch, val);
        }
    }

#ifdef CONFIG_PCIEASPM
    {
	extern int pcie_aspm_policy(int);
        if (pcie_aspm_policy(ch) == 3)// POLICY_POWERSUPERSAVE
        {
            dw_pcie_writel_phy(dw_plat_pcie->pci, GENERAL_CORE_CTRL_OFF,\
                    dw_pcie_readl_phy(dw_plat_pcie->pci,GENERAL_CORE_CTRL_OFF) |APP_CLK_REQ_N);
        }
    }
#endif

    //set PERSTN_MASK
    val = dw_pcie_readl_phy(dw_plat_pcie->pci, GENERAL_SIGNAL_CTRL_OFF);
    val |= PERSTN_MASK;
    dw_pcie_writel_phy(dw_plat_pcie->pci, GENERAL_SIGNAL_CTRL_OFF, val);

    //set type as RC
    dw_pcie_set_Mode(dw_plat_pcie->pci, DW_PCIE_AS_RC);

    //turn on axi fabric pcie clock/reset
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_AXI2AVB_PCIE0_CLK]);
    ret |= clk_prepare_enable(dw_plat_pcie->pclk[E_PLAT_AXI2AVB_PCIE1_CLK]);

    if(reset_control_status(dw_plat_pcie->prst[E_PLAT_AXI2AVB_PCI_RST]))
        ret |= reset_control_deassert(dw_plat_pcie->prst[E_PLAT_AXI2AVB_PCI_RST]);

    if(reset_control_status(dw_plat_pcie->prst[E_PLAT_AXI2AVB_PCI_I0_RST]))
        ret |= reset_control_deassert(dw_plat_pcie->prst[E_PLAT_AXI2AVB_PCI_I0_RST]);

    if (!dw_plat_pcie->pci->pp.handover) {
    t1 = get_cycles();
    do
    {
        msleep(1);
        dw_pcie_get_phy_status(dw_plat_pcie->pci, dw_plat_pcie->lanes, &status);

        max_try++;
        if((max_try > PHY_READY_MAX_TRY) && (status == DW_PCIE_PHY_IDLE))
        {
            dw_plat_suspend_sys(dw_plat_pcie);
            return -1;
        }
    } while(status == DW_PCIE_PHY_IDLE);
    t1 = get_cycles() - t1;
#ifdef CONFIG_ARCH_CXD900XX
    t1 = udif_cycles_to_usecs(t1);
#endif
    printk(KERN_ERR "PCI%d: PHY ready in %lld\n", ch, t1);
    }

    /* set VID/DID */
    dw_pcie_dbi_ro_wr_en(dw_plat_pcie->pci);
    dw_pcie_writew_dbi(dw_plat_pcie->pci, PCI_VENDOR_ID, PCI_VENDOR_ID_SONY);
    dw_pcie_writew_dbi(dw_plat_pcie->pci, PCI_DEVICE_ID, PCI_DEVICE_ID_CXD90057);
    dw_pcie_dbi_ro_wr_dis(dw_plat_pcie->pci);

    { /* set AXI link timeout */
	    int tmo = pcie_dwc_axi_tmo(ch);
	    if (0 <= tmo  &&  tmo <= 0x1ff) {
		    dw_pcie_writel_dbi(dw_plat_pcie->pci, AMBA_LINK_TIMEOUT_OFF, tmo);
		    printk(KERN_INFO "PCI%d: AXITMO=0x%x\n", ch, tmo);
	    }
    }
    return ret;
}


#if 0
static irqreturn_t dw_plat_prsnt_intr(int irq, void *dev_id)
{
	struct pcie_port *pp = dev_id;
	cycles_t now;
	unsigned long flags;
	char buf[64];

	now = get_cycles();
	spin_lock_irqsave(&pp->prsnt_lock, flags);
	if (pp->prsnt_negate < now) {
		pp->prsnt_negate = now;
	}
	spin_unlock_irqrestore(&pp->prsnt_lock, flags);

	scnprintf(buf, sizeof buf, "PCI:/PRSNT(IRQ%d)=H", irq);
	BOOT_TIME_ADD1(buf);
	return IRQ_HANDLED;
}
#endif

static void dw_plat_setup_prsnt(struct pcie_port *pp,
				struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	int ch;
#if 0
	int ret;
#endif

	spin_lock_init(&pp->prsnt_lock);
	pp->prsnt = -1;
	pp->prsnt_negate = 0;
	pp->prsnt_assert = 0;
	ch = of_get_pci_domain_nr(dev->of_node);
	pp->prsnt_irq = pcie_prsnt_irq(ch);
	if (pp->prsnt_irq < 0)
		return;
#if 0
	ret = devm_request_irq(dev, pp->prsnt_irq, dw_plat_prsnt_intr,
			       IRQF_NO_THREAD,
			       "pcie_prsnt", pp);
	if (ret) {
		dev_err(dev, "failed to requst PRSNT IRQ\n");
		return;
	}
#endif
	printk(KERN_INFO "PCI%d: /PRSNT IRQ=%d\n", ch, pp->prsnt_irq);
}



static int dw_plat_suspend_sys(struct dw_plat_pcie* dw_plat_pcie)
{
    int ret = 0;
    u32 val;

    //assert reset
    ret |= reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_PWRUP_RST]);
    ret |= reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_BUTTON_RST]);
    ret |= reset_control_assert(dw_plat_pcie->prst[E_PLAT_PCIE_PE_RST]);

    //CLKREQ# Hi-Z
    val = dw_pcie_readl_phy(dw_plat_pcie->pci, GENERAL_SIGNAL_CTRL_OFF);
    val |= CLKREQ_FW;
    dw_pcie_writel_phy(dw_plat_pcie->pci, GENERAL_SIGNAL_CTRL_OFF, val);

    //set IDDQ=1
    val = dw_pcie_readl_phy(dw_plat_pcie->pci, PHY_CONFIG_0X18_OFF);
    val |= PHY_POWERDOWN;
    dw_pcie_writel_phy(dw_plat_pcie->pci, PHY_CONFIG_0X18_OFF, val);

    wmb();
    udelay(1);

    //gated clock
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_PCIE_APB_PCLK]);
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_PCIE_DBI_ACLK]);
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_PCIE_SLV_ACLK]);
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_PCIE_MSTR_ACK]);
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_PCIE_AUX_CLK]);
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_AXI2AVB_PCIE0_CLK]);
    clk_disable_unprepare(dw_plat_pcie->pclk[E_PLAT_AXI2AVB_PCIE1_CLK]);

    return ret;
}
#endif
#define SLOT_POWER_LIMIT	0x30
static void dw_plat_pcie_fixup_bridge(struct pci_dev *dev)
{
	struct pcie_port *pp = dev->bus->sysdata;
	struct dw_pcie *dwp = to_dw_pcie_from_pp(pp);
	int ch = pci_domain_nr(dev->bus);
	u32 slot_cap;
	u16 caps_reg;

	if (pci_pcie_type(dev) != PCI_EXP_TYPE_ROOT_PORT)
		return;

	dw_pcie_dbi_ro_wr_en(dwp);
	caps_reg = pcie_caps_reg(dev) | PCI_EXP_FLAGS_SLOT;
	pcie_capability_write_word(dev, PCI_EXP_FLAGS, caps_reg);
	dev->pcie_flags_reg = caps_reg;

	pcie_capability_read_dword(dev, PCI_EXP_SLTCAP, &slot_cap);
	slot_cap = slot_cap & (~PCI_EXP_SLTCAP_SPLV);
	slot_cap = slot_cap | (SLOT_POWER_LIMIT << 7);
	slot_cap &= ~PCI_EXP_SLTCAP_PSN;
	slot_cap |= (pci_domain_nr(dev->bus) + 1) << PCI_EXP_SLTCAP_PSN_SHIFT;
	pcie_capability_write_dword(dev, PCI_EXP_SLTCAP, slot_cap);
	dw_pcie_dbi_ro_wr_dis(dwp);

	{ /* set completion timeout */
		int tmo;
		u32 val;

		tmo = pcie_dwc_tmo(ch);
		if (0 <= tmo  &&  tmo <= 0x1f) {
			pcie_capability_read_dword(dev, PCI_EXP_DEVCTL2, &val);
			val &= ~0x1fU;
			val |= tmo;
			pcie_capability_write_dword(dev, PCI_EXP_DEVCTL2, val);
			printk(KERN_INFO "PCI%d: TMO=0x%x\n", ch, tmo);
		}
	}

	if (pcibios_enable_device(dev, ~0) < 0) {
		pr_err("PCI: synopsys device enable failed\n");
		return;
	}
}
DECLARE_PCI_FIXUP_EARLY(PCI_ANY_ID, PCI_ANY_ID, dw_plat_pcie_fixup_bridge);

static irqreturn_t dw_plat_pcie_msi_irq_handler(int irq, void *arg)
{
	struct pcie_port *pp = arg;

	return dw_handle_msi_irq(pp);
}

static int dw_plat_pcie_host_init(struct pcie_port *pp)
{
    struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
    int ch = of_get_pci_domain_nr(pci->dev->of_node);

#if 0
    dw_pcie_RC_Assert_Reset(pci);
#else
    pcie_perst_delay(ch);
    dw_pcie_set_PERST(pci, 0); // negate PERST
#endif
    dw_pcie_setup_rc(pp);
    if (of_property_read_bool(pci->dev->of_node, "cxd90057,pci32")) {
        if ((PHYS_OFFSET >> 32) != 0ULL) {
            /* map PCI:00000000 to AXI:PHYS_OFFSET */
            dw_pcie_prog_ib_atu2(pci, PCIE_ATU_REGION_INDEX0,
                    0x00000000ULL, 0x100000000ULL,
                    PHYS_OFFSET & ~0xFFFFFFFFULL,
                    DW_PCIE_AS_MEM);
        }
    }
    if (!pp->handover) {
    dw_pcie_set_LTSSM_en(pci, 1);
    dw_pcie_wait_for_link(pci);
    }

    if (IS_ENABLED(CONFIG_PCI_MSI))
        dw_pcie_msi_init(pp);

    pcie_dwc_host_notify(ch, 1);
    return 0;
}

static const struct dw_pcie_host_ops dw_plat_pcie_host_ops = {
	.host_init = dw_plat_pcie_host_init,
};

static int dw_plat_add_pcie_port(struct pcie_port *pp,
				 struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	int ret;

	pp->irq = platform_get_irq(pdev, 1);
	if (pp->irq < 0)
		return pp->irq;

	if (IS_ENABLED(CONFIG_PCI_MSI)) {
		pp->msi_irq = platform_get_irq(pdev, 0);
		if (pp->msi_irq < 0)
			return pp->msi_irq;

		ret = devm_request_irq(dev, pp->msi_irq,
					dw_plat_pcie_msi_irq_handler,
					IRQF_SHARED | IRQF_NO_THREAD,
					"dw-plat-pcie-msi", pp);
		if (ret) {
			dev_err(dev, "failed to request MSI IRQ\n");
			return ret;
		}
	}
	dw_plat_setup_prsnt(pp, pdev);

	pp->root_bus_nr = -1;
	pp->root_bus = NULL;
	pp->ops = &dw_plat_pcie_host_ops;

	ret = dw_pcie_host_init(pp);
	if (ret) {
		dev_err(dev, "failed to initialize host\n");
		return ret;
	}

	return 0;
}


extern u32 dw_pcie_readl_phy(struct dw_pcie *pci, u32 reg);
static int dw_plat_pcie_linkup(struct dw_pcie *pcie)
{
    if(dw_pcie_readl_phy(pcie, DEBUG_SIGNALS) & RDLH_LINK_UP)
        return 1;
    else
        return  0;
}

static const struct dw_pcie_ops dw_pcie_ops = {
	.link_up = dw_plat_pcie_linkup,
};

static int dw_plat_pcie_probe(struct platform_device *pdev)
{
    struct device *dev = &pdev->dev;
    struct dw_plat_pcie *dw_plat_pcie;
    struct dw_pcie *pci;
    struct resource *res;  /* Resource from DT */
    int ret;
#ifndef CONFIG_ARCH_CXD900XX_FPGA
    u8 i;
    struct device_node *np = dev->of_node;
#endif

    dw_plat_pcie = devm_kzalloc(dev, sizeof(*dw_plat_pcie), GFP_KERNEL);
    if (!dw_plat_pcie)
        return -ENOMEM;

    pci = devm_kzalloc(dev, sizeof(*pci), GFP_KERNEL);
    if (!pci)
        return -ENOMEM;

    pci->dev = dev;
    pci->ops = &dw_pcie_ops;

    dw_plat_pcie->pci = pci;

    res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
    pci->dbi_base = devm_ioremap_resource(dev, res);
    if (IS_ERR(pci->dbi_base))
        return PTR_ERR(pci->dbi_base);

    //get phy base for side band signal setting
    res = platform_get_resource_byname(pdev, IORESOURCE_MEM, "phy_reg");
    pci->phy_base = devm_ioremap_resource(dev, res);
    if (IS_ERR(pci->phy_base))
        return PTR_ERR(pci->phy_base);

#ifndef CONFIG_ARCH_CXD900XX_FPGA
    for(i=0;i<E_PLAT_RST_TOTAL;i++)
    {
        if(i < E_PLAT_AXI2AVB_PCI_RST)
            dw_plat_pcie->prst[i]=devm_reset_control_get(dev, rst_ctrl_name[i]);
        else
            dw_plat_pcie->prst[i]=devm_reset_control_get_shared(dev, rst_ctrl_name[i]);

        if(IS_ERR(dw_plat_pcie->prst[i]))
        {
            dev_err(dev,"Cannot get %s\n",rst_ctrl_name[i] );
            return -ENODEV;
        }
    }


    for(i=0;i<E_PLAT_CLK_TOTAL;i++)
    {
        dw_plat_pcie->pclk[i]=devm_clk_get(dev, clk_gate_name[i]);
        if(IS_ERR(dw_plat_pcie->pclk[i]))
        {
            dev_err(dev,"Cannot get %s\n",clk_gate_name[i] );
            return -ENODEV;
        }
    }

    ret = of_property_read_u32(np, "num-lanes", &dw_plat_pcie->lanes);
    if (ret)
        dw_plat_pcie->lanes = 0;

    ret = of_property_read_u32(np, "firm-id", &dw_plat_pcie->firm);
    if (ret || dw_plat_pcie->firm > N_FIRM) {
	dw_plat_pcie->firm = 0; /* no firm */
    }

    if(dw_plat_setup_sys(dw_plat_pcie))
        return -EPERM;
#else
    dw_pcie_set_Mode(pci, DW_PCIE_AS_RC);
#endif

    platform_set_drvdata(pdev, dw_plat_pcie);

    ret = dw_plat_add_pcie_port(&pci->pp, pdev);
    if (ret < 0)
        return ret;

#ifdef CONFIG_PCIE_DW_DEBUG
    dw_pcie_debugfs_init(pci);
#endif

    return 0;
}

static const struct of_device_id dw_plat_pcie_of_match[] = {
	{ .compatible = "snps,dw-pcie", },
	{},
};

static int dw_plat_pcie_remove(struct platform_device *pdev)
{
    struct device *dev = &pdev->dev;
    struct dw_plat_pcie *dw_plat_pcie;
    struct pcie_port *pp ;
    int ret = 0;

    dw_plat_pcie = platform_get_drvdata(pdev);
    pp = &dw_plat_pcie->pci->pp;

    pcie_dwc_host_notify(pci_domain_nr(pp->root_bus), 0);

    ret = dw_pcie_Remove_RC(dw_plat_pcie->pci);

    if (IS_ENABLED(CONFIG_PCI_MSI))
    {
        devm_free_irq(dev, pp->msi_irq, pp);
        free_pages(pp->msi_data, 0);
    }

#ifdef CONFIG_PCIE_DW_DEBUG
    dw_pcie_debugfs_exit(dw_plat_pcie->pci);
#endif

#ifndef CONFIG_ARCH_CXD900XX_FPGA
    dw_plat_suspend_sys(dw_plat_pcie);
#endif

    printk("%s %d: Remove PCIe RC Complete\n", __FUNCTION__, __LINE__);

    return ret;
}

static int dw_plat_pcie_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dw_plat_pcie *dw_plat_pcie;
	int ch;

	dw_plat_pcie = platform_get_drvdata(pdev);
	ch = pci_domain_nr(dw_plat_pcie->pci->pp.root_bus);
	if (!ch)
		return 0;

	printk(KERN_ERR "PCI%d:ERROR:please turn off before suspend\n", ch);
	BUG();
}

static struct platform_driver dw_plat_pcie_driver = {
    .driver = {
        .name	= "dw-pcie",
        .of_match_table = dw_plat_pcie_of_match,
        .suppress_bind_attrs = true,
    },
    .probe = dw_plat_pcie_probe,
    .remove = dw_plat_pcie_remove,
    .suspend = dw_plat_pcie_suspend,
};
builtin_platform_driver(dw_plat_pcie_driver);

int dw_plat_pcie_lanes_dynamic_change(struct pci_dev *dev, unsigned int target_lanes)
{
    struct pcie_port *pp;
    struct dw_pcie *pci;
    int err;

    if (!dev)
        return -ENODEV;

    pp = dev->bus->sysdata;
    pci = to_dw_pcie_from_pp(pp);

    err = dw_pcie_lanes_dynamic_change(pci, target_lanes);
    if (err) {
        u32 state = dw_pcie_readl_phy(pci, PM_LTSSM_STATE);
        printk("%s %d: err=%d, PM_LTSSM_STATE=0x%08X\n", __FUNCTION__, __LINE__, err, state);
    }

    return err;
}

int dw_plat_pcie_negotiated_link_width(struct pci_dev *dev)
{
    struct pcie_port *pp;
    struct dw_pcie *pci;
    u32 pos;
    u16 lnksta, link_width;

    if (!dev)
        return -ENODEV;

    pp = dev->bus->sysdata;
    pci = to_dw_pcie_from_pp(pp);

    pos = dw_pcie_find_capability(pci, PCI_CAP_ID_EXP);
    if (!pos)
        return -ENXIO;

    /* Link Status */
    pos += PCI_EXP_LNKSTA;
    lnksta = dw_pcie_readw_dbi(pci, pos);
    /* Negotiated Link Width */
    link_width = ((lnksta & PCI_EXP_LNKSTA_NLW) >> PCI_EXP_LNKSTA_NLW_SHIFT);
    if (link_width == 0)
        link_width = -EIO;

    return link_width;
}

#define PCIE_AER_POS 0x100

void dw_plat_pcie_err_clr(struct dw_pcie *pci)
{
	dw_pcie_writel_dbi(pci, PCIE_AER_POS+PCI_ERR_UNCOR_STATUS, ~0U);
}

int dw_plat_pcie_err_stat(struct dw_pcie *pci)
{
	u32 stat;

	stat = dw_pcie_readl_dbi(pci, PCIE_AER_POS+PCI_ERR_UNCOR_STATUS);
	if (stat) {
		int ch = of_get_pci_domain_nr(pci->dev->of_node);
		char buf[64];
		dw_pcie_writel_dbi(pci, PCIE_AER_POS+PCI_ERR_UNCOR_STATUS, stat);
		scnprintf(buf, sizeof buf, "PCI%d:%s:0x%x\n", ch, __func__, stat);
		BOOT_TIME_ADD1(buf);
		return -1;
	}
	return 0;
}

/* called with IRQ disabled */
static int __dw_plat_check_perst(struct pcie_port *pp, int ch)
{
	int perst;

	/* read /PERST pin */
	perst = pcie_get_perst(ch);
	if (perst < 0) { /* not supported */
		return 0;
	}
	return perst;
}

/* called with IRQ disabled */
static int __dw_plat_check_prsnt(struct pcie_port *pp, int ch, unsigned long *flags)
{
	cycles_t tmo, now;
	int done, prsnt, pending, linkdn;
	char buf[64];

	smp_rmb();
	if (!pp->prsnt)
		return 0;

	local_irq_restore(*flags);
	scnprintf(buf, sizeof buf, "PCI%d:/PRSNT IRQ pending", ch);
	local_irq_save(*flags);
	tmo = get_cycles() + pcie_prsnt_tmo(ch);
	done = 0;
	do {
		local_irq_restore(*flags);
		spin_lock_irqsave(&pp->prsnt_lock, *flags);

		if (!pp->prsnt) {
			spin_unlock(&pp->prsnt_lock);
			/* IRQ is disabled */
			return 0;
		}

		/* check IRQ */
		pending = (pp->prsnt_irq < 0) ? 0 : GIC_IRQ_PENDING(pp->prsnt_irq);
		if (pending) {
			now = get_cycles();
			if (pp->prsnt_negate < now) {
				pp->prsnt_negate = now;
				pp->prsnt_assert = now;
			}
			gic_clearirq(pp->prsnt_irq);
			BOOT_TIME_ADD1(buf);
		}

		/* read /PRSNT pin */
		prsnt = pcie_prsnt(ch);
		now = get_cycles();
		if (prsnt < 0) { /* not supported */
			done = 1;
		} else if (!prsnt) { /* negate */
			if (pp->prsnt_negate < now)
				pp->prsnt_negate = now;
		} else { /* assert */
			if (pp->prsnt_assert < now)
				pp->prsnt_assert = now;
		}

		/* judge */
		if (now > pp->prsnt_assert + pcie_prsnt_high(ch)) {
			/* ___HHH_____HHHHHHHHHHHHHHHHHHHHHHxxxxx
			 *           |<--pcie_prsnt_high-->|
			 *           |                     |
			 *      prsnt_assert            now,presnt_negate
			 */
			done = 1;
		}
		if (now > pp->prsnt_negate + pcie_prsnt_low1(ch)) {
			/* ___HHH__HHHH_____________________xxxxx
			 *            |<--pcie_prsnt_low1->|
			 *            |                    |
			 *       prsnt_negate           now,prsnt_assert
			 */
			done = 1;
		}
		if (now >= tmo &&
		    now > pp->prsnt_negate + pcie_prsnt_low2(ch)) {
			/* ___HHH__HHHH______________________xxxxx
			 *            |<--pcie_prsnt_low2-->|
			 *            |                     |
			 *       prsnt_negate           now,prsnt_assert
			 */
			done = 1;
		}

		spin_unlock(&pp->prsnt_lock);
	} while (!done && get_cycles() < tmo);
	/* IRQ is disabled */

	if (prsnt < 0)
		return 1;

	linkdn = 0;
	if (done && prsnt) {
		struct dw_pcie *pci = to_dw_pcie_from_pp(pp);
		if (dw_plat_pcie_linkup(pci))
			return 1;
		linkdn = 1;
	}

	pp->prsnt = 0;
	smp_wmb();
	local_irq_restore(*flags);
	if (linkdn)
		scnprintf(buf, sizeof buf, "PCI%d:LINKDN", ch);
	else
		scnprintf(buf, sizeof buf, "PCI%d:/PRSNT=H", ch);
	BOOT_TIME_ADD1(buf);
	local_irq_save(*flags);
	return 0;
}

int dw_plat_check_prsnt(struct pcie_port *pp, struct pci_bus *bus)
{
	int ch = pci_domain_nr(bus);
	int ret, reset;
	unsigned long flags;

	local_irq_save(flags);
	ret = __dw_plat_check_prsnt(pp, ch, &flags);
	reset = __dw_plat_check_perst(pp, ch);
	local_irq_restore(flags);
	return ret && !reset;
}

/* called with IRQ disabled */
static inline int pcie_check_prsnt(struct pci_dev *pdev, unsigned long *flags)
{
	int ch = pci_domain_nr(pdev->bus);
	int prsnt, reset;

	if (!pcie_has_prsnt(ch))
		prsnt = 1;
	else
		prsnt = __dw_plat_check_prsnt(pdev->bus->sysdata, ch, flags);

	reset = __dw_plat_check_perst(pdev->bus->sysdata, ch);

	return prsnt && !reset;
}

int pcie_write32(struct pci_dev *pdev, volatile void __iomem *addr, u32 val)
{
	int ret = -1;
	unsigned long flags;

	local_irq_save(flags);
	if (pcie_check_prsnt(pdev, &flags)) {
		writel(val, addr);
		ret = 0;
	}
	local_irq_restore(flags);
	return ret;
}

int pcie_read32(struct pci_dev *pdev, volatile void __iomem *addr, u32 *val)
{
	int ret = -1;
	unsigned long flags;
	unsigned long long t1 = 0, t2 = 0, dt;

	local_irq_save(flags);
	if (pcie_check_prsnt(pdev, &flags)) {
		struct dw_pcie *pci = to_dw_pcie_from_pp(pdev->bus->sysdata);
		dw_plat_pcie_err_clr(pci);

		t1 = udif_read_freerun();
		*val = readl(addr);
		t2 = udif_read_freerun();

		if (dw_plat_pcie_err_stat(pci) < 0) {
			*val = ~0U;
		} else {
			ret = 0;
		}
	}
	local_irq_restore(flags);
	dt = udif_cycles_to_usecs(t2 - t1);
	if (dt >= PCIE_STALL_THRESHOLD) {
		char buf[64];
		scnprintf(buf,sizeof buf,"PCIe:STALL:%s:%llu",__func__,dt);
		BOOT_TIME_ADD1(buf);
		printk(KERN_ERR "%s\n", buf);
		dump_stack();
	}
	return ret;
}

int pcie_read64(struct pci_dev *pdev, volatile void __iomem *addr, u64 *val)
{
	int ret = -1;
	unsigned long flags;
	unsigned long long t1 = 0, t2 = 0, dt;

	local_irq_save(flags);
	if (pcie_check_prsnt(pdev, &flags)) {
		struct dw_pcie *pci = to_dw_pcie_from_pp(pdev->bus->sysdata);
		dw_plat_pcie_err_clr(pci);

		t1 = udif_read_freerun();
		*val = readq(addr);
		t2 = udif_read_freerun();

		if (dw_plat_pcie_err_stat(pci) < 0) {
			*val = ~0UL;
		} else {
			ret = 0;
		}
	}
	local_irq_restore(flags);
	dt = udif_cycles_to_usecs(t2 - t1);
	if (dt >= PCIE_STALL_THRESHOLD) {
		char buf[64];
		scnprintf(buf,sizeof buf,"PCIe:STALL:%s:%llu",__func__,dt);
		BOOT_TIME_ADD1(buf);
		printk(KERN_ERR "%s\n", buf);
		dump_stack();
	}
	return ret;
}
