/*
 * drivers/usb/f_usb/udif_usb_otg.c
 *
 * Copyright (C) 2011-2012 FUJITSU SEMICONDUCTOR LIMITED
 * Copyright 2018 Sony Imaging Products and Solutions Incorporated.
 *
 * ALL RIGHTS RESERVED, COPYRIGHT (C) SOCIONEXT INC. 2015
 * LICENSED MATERIAL - PROGRAM PROPERTY OF SOCIONEXT INC.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*--------------------------------------------------------------------------*
 * drivers/usb/scd/scd_device.c
 *
 * OTG and, Platform bus driver operations
 *
 * Copyright 2011 Sony Corporation
 *
 * This file is part of the HS-OTG Controller Driver SCD.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *---------------------------------------------------------------------------*/

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/jiffies.h>
#include <linux/gpio/gpio.h>
#include <mach/irqs.h>
#include <mach/gic_export.h>

#include <linux/udif/module.h>
#include <linux/udif/device.h>
#include <linux/udif/types.h>
#include <linux/udif/irq.h>
#include <linux/udif/io.h>
#include <linux/udif/spinlock.h>
#include <linux/udif/list.h>
#include <linux/udif/tasklet.h>
#include <linux/udif/malloc.h>
#include <linux/udif/mutex.h>
#include <linux/udif/delay.h>
#include <linux/udif/proc.h>
#include <linux/usb/f_usb/usb_otg_extcon.h>

#include "linux/usb/f_usb/usb_otg_notify.h"
#include "usb_otg_reg.h"
#include "udif_usb_otg.h"
#include "udif_usb_otg_if.h"
#include "udif_usb_otg_gpioint.h"

//#define FUSB_FUNC_TRACE					/**<  */

#ifdef FUSB_FUNC_TRACE
#define fusb_func_trace(b) printk("CALL %s %s(%d)\n",(b),__FUNCTION__,__LINE__)
#define fusb_print_dbg(fmt, arg...) printk(KERN_DEBUG fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#define fusb_print_err(fmt, arg...) printk(KERN_ERR fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#else
#define fusb_func_trace(b)
#define fusb_print_dbg(fmt, arg...)
/*#define fusb_print_err(fmt, arg...)*/
#define fusb_print_err(fmt, arg...) printk(KERN_ERR fmt " %s(%d)\n", ##arg, __FUNCTION__,__LINE__)
#endif

#ifdef NDEBUG
#define fusb_assert_NULL(b)
#else /* #ifdef NDEBUG */
#define fusb_assert_NULL(b)								\
	if (!(b))									\
	{										\
		printk(KERN_ERR " %s(%d) 0x%x *** ASSERT \n",__FILE__,__LINE__,(b));	\
		BUG();									\
	}				 /**<  */
#endif


/**
 * @brief usb_str
 *
 *
 */
static char* usb_str = "fusb_otg";
UDIF_MODULE_PARAM(usb_str, charp, 0664U);

UDIF_IDS(fusb_notify_otg_ids) = {
	UDIF_ID(UDIF_ID_FUSB_OTG, 0xFFU)	/* 0b:OTG Register I/O, 1b:USB_VBUS, 2b:USB_ID */
};

static UDIF_DRIVER_OPS fusb_notify_otg_cb = {
	.init       = fusb_notify_otg_init,
	.exit       = fusb_notify_otg_exit,
	.probe      = fusb_notify_otg_probe,
	.remove     = fusb_notify_otg_remove,
	.suspend    = fusb_notify_otg_suspend,
	.resume     = fusb_notify_otg_resume,
	.shutdown   = fusb_notify_otg_shutdown,
};

UDIF_MODULE(fusb_otg, "fusb_otg", "1.0", fusb_notify_otg_cb, fusb_notify_otg_ids, NULL, NULL);

static struct fusb_notify_control *g_fusb_notify_control = NULL;

static UDIF_LIST_HEAD(fusb_anchor);

static UDIF_DECLARE_SPINLOCK(fusb_lock);

static int fusb_create_proc_flag = FUSB_OTG_FLAG_OFF;

#ifdef CONFIG_USB_FUSB_OTG_TEST
void *fusb_test_get_notify_control(void)
{
	return (void*)g_fusb_notify_control;
}
EXPORT_SYMBOL_GPL(fusb_test_get_notify_control);

void fusb_test_signal_event(UDIF_ULONG data)
{
	fusb_signal_event(data);
}
EXPORT_SYMBOL_GPL(fusb_test_signal_event);

irqreturn_t fusb_test_interrupt_usb_id(int irq, void *p)
{
	return fusb_interrupt_usb_id(irq, p);
}
EXPORT_SYMBOL_GPL(fusb_test_interrupt_usb_id);

irqreturn_t fusb_test_interrupt_usb_vbus(int irq, void *p)
{
	return fusb_interrupt_usb_vbus(irq, p);
}
EXPORT_SYMBOL_GPL(fusb_test_interrupt_usb_vbus);

#endif

static int fusb_read_id(void);
static int fusb_read_vbus0(void);
static int fusb_read_vbus1(void);
static int fusb_read_proc(UDIF_PROC_READ *proc)
{
	int len = 0;
	int err;
	struct fusb_notify_control *fusb;
	bool state;

	if (proc == NULL) {
		fusb_print_err("argument proc is set NULL");
		return UDIF_ERR_PAR;
	}

	len += udif_proc_setbuf(proc, len, "USB Controller Driver Ver : %d\n", FUSB_VERSION);

	len += udif_proc_setbuf(proc, len, "  internal latch:\n");
	fusb = g_fusb_notify_control;
	if (fusb) {
		if (FUSB_OTG_FLAG_ON != fusb->start_comp_flag) {
			len += udif_proc_setbuf(proc, len, "    not started\n");
		}
		len += udif_proc_setbuf(proc, len, "    usb_id=%d\n", fusb->cid);
		len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT0]  =%d\n", fusb->vbus[USB_OTG_VBUS_PORT0]);
		len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT1]  =%d\n", fusb->vbus[USB_OTG_VBUS_PORT1]);
	}
	else {
		len += udif_proc_setbuf(proc, len, "    not started\n");
	}
	len += udif_proc_setbuf(proc, len, "  readouts:\n");
	len += udif_proc_setbuf(proc, len, "    usb_id=%d\n", fusb_read_id());
	len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT0]  =%d\n", fusb_read_vbus0());
	len += udif_proc_setbuf(proc, len, "    vbus[USB_OTG_VBUS_PORT1]  =%d\n", fusb_read_vbus1());

	err = fusb_otg_get_drd_state(&state);
	if (err)
		len += udif_proc_setbuf(proc, len, "extcon get_drd_state err=%d\n", err);
	else
		len += udif_proc_setbuf(proc, len, "extcon drd=%d(host=%d)\n", state, true);

	err = fusb_otg_is_orientation_enabled();
	if (0 >= err) {
		len += udif_proc_setbuf(proc, len, "extcon orientation is not enabled err=%d\n", err);
		goto bail;
	}
	if (fusb_otg_get_orientation(&state))
		len += udif_proc_setbuf(proc, len, "extcon get_orientation err\n");
	else
		len += udif_proc_setbuf(proc, len, "extcon typec orientation=%d(flipped=%d)\n", state, true);

	err = fusb_otg_is_ss_operation_enabled();
	if (0 >= err) {
		len += udif_proc_setbuf(proc, len, "extcon ss_operation is not enabled err=%d\n", err);
		goto bail;
	}
	if (fusb_otg_get_ss_operation(&state))
		len += udif_proc_setbuf(proc, len, "extcon get_ss_operation err\n");
	else
		len += udif_proc_setbuf(proc, len, "extcon typec ss_operation=%d(enable=%d)\n", state, true);

bail:
	udif_proc_setend(proc);
	return len;
}

extern void fusb_otg_flip_drd_state(void);
extern void fusb_otg_flip_orientation(void);
static UDIF_INT fusb_write_proc(UDIF_PROC_WRITE *proc)
{
	char buf[32];
	UDIF_SIZE cnt = 0;

	/* change proc command */
	cnt = udif_proc_getbuf(proc, buf, sizeof(buf) - 1);
	
	if (cnt > 0) {
		buf[cnt-1] = '\0';
		if (strncmp(buf, "drd_flip", cnt) == 0) {
		    fusb_otg_flip_drd_state();
		}
		else if (strncmp(buf, "typec_flip", cnt) == 0) {
		    fusb_otg_flip_orientation();
		}
		else if (strncmp(buf, "ss_op", cnt) == 0) {
		    fusb_otg_flip_ss_operation();
		}
		else if (strncmp(buf, "help", cnt) == 0) {
		    printk(KERN_INFO "commands: help, drd_flip, typec_flip, ss_op\n");
		}
	}

	return udif_proc_bufsize(proc);
}

/**
 * @brief fusb_proc
 *
 *
 */
static UDIF_PROC fusb_proc = {
	.name  = FUSB_PROC_NAME,
	.read  = fusb_read_proc,
	.write = fusb_write_proc,
};

static UDIF_ERR fusb_notify_otg_init(UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	UDIF_ERR ret;
	UDIF_UINT size;

	fusb_func_trace("START");

	size = sizeof(struct fusb_notify_control);
	fusb = udif_kmalloc("fusb_notiry", size, 0);
	if (fusb == NULL) {
		fusb_print_err("udif_kmalloc is failed");
		return UDIF_ERR_NOMEM;
	}

	memset(fusb, 0, size);
	fusb->otg_regs      = NULL;
	udif_spin_lock_init(&fusb->lock);
	udif_tasklet_init(&fusb->event_tasklet, fusb_signal_event, (UDIF_ULONG)fusb);
	udif_list_head_init(&fusb->event_anchor);
	fusb->cid           = FUSB_B_DEVICE;
	fusb->vbus[USB_OTG_VBUS_PORT0] = FUSB_VBUS_OFF;
	fusb->vbus[USB_OTG_VBUS_PORT1] = FUSB_VBUS_OFF;
	fusb->gadget_con    = FUSB_EVENT_DISCONNECT;
	fusb->init_comp_flag = FUSB_OTG_FLAG_ON;
	fusb->start_comp_flag = FUSB_OTG_FLAG_OFF;

	init_timer(&fusb->con_timer);
	fusb->con_timer.function    = fusb_con_timer_func;

	g_fusb_notify_control = fusb;

	if ((ret = udif_create_proc(&fusb_proc)) != UDIF_ERR_OK) {
		fusb_print_err("udif_create_proc is failed");
		return ret;
	}
	fusb_create_proc_flag = FUSB_OTG_FLAG_ON;

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_exit(UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	UDIF_ERR ret;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb_create_proc_flag == FUSB_OTG_FLAG_ON) {
		if ((ret = udif_remove_proc(&fusb_proc)) != UDIF_ERR_OK) {
			fusb_print_err("udif_remove_proc is failed");
		} else {
			fusb_create_proc_flag = FUSB_OTG_FLAG_OFF;
		}
	}

	if (g_fusb_notify_control == NULL) {
		return UDIF_ERR_OK;
	}

	g_fusb_notify_control = NULL;

	udif_tasklet_kill(&fusb->event_tasklet);
	udif_kfree((UDIF_VP)fusb);

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static int fusb_read_vbus0(void)
{
	 unsigned int val;
	 int ret;

	ret = gpio_get_data_bit(FUSB_VBUS_GPIO_PORT, FUSB_VBUS0_GPIO_BIT, &val);
	if (ret) fusb_print_err("vbus gpio_get_data_bit is failed");
	return val;
}

static int fusb_read_vbus1(void)
{
	 unsigned int val;
	 int ret;

	ret = gpio_get_data_bit(FUSB_VBUS_GPIO_PORT, FUSB_VBUS1_GPIO_BIT, &val);
	if (ret) fusb_print_err("vbus gpio_get_data_bit is failed");
	return val;
}

static int fusb_read_id(void)
{
	unsigned int val;
#ifdef FUSB_USE_USB_ID
	int ret;

	ret = gpio_get_data_bit(FUSB_ID_GPIO_PORT, FUSB_ID_GPIO_BIT, &val);
	if (ret) fusb_print_err("id gpio_get_data_bit is failed");
#else
	val = 1; /* means CID_B */
#endif
	return val;
}

static int fusb_configure_gpio(unsigned char port, unsigned int bit, irq_handler_t hdlr, char *name)
{
	int ret;
	int irq;
	unsigned long flags = IRQF_TRIGGER_RISING;

	ret = gpio_set_mode_port(port, 1<<bit, GPIOMODE_GPIO | GPIOMODE_INPUT);
	if (ret) {
		fusb_print_err("%s gpio_set_mode_port (gpio input) err=%d", name, ret);
		return -1;
	}

#ifdef CONFIG_ARCH_CXD900XX_FPGA
	irq = gpiopin_to_irq(GPIO_INT, bit);
#else
	irq = gpiopin_to_irq(port, bit);
#endif
	if (0 >= irq) {
		fusb_print_err("%s gpiopin_to_irq fail", name);
		return -1;
	}

	gic_clearirq(irq);
	ret = request_irq(irq, hdlr, flags | IRQF_NO_THREAD, name, NULL);
	if (ret) {
		fusb_print_err("%s request_irq fail err=%d, irq=%d", name, ret, irq);
		return -1;
	}

	disable_irq(irq);

	ret = gpio_set_mode_port(port, 1<<bit, GPIOMODE_PERIPHERAL | GPIOMODE_INPUT);
	if (ret) {
		fusb_print_err("%s gpio_set_mode_port (peri input) err=%d", name, ret);
		return -1;
	}

	/* configure gpiointxxx registers here if necessary */
	fusb_setup_gpioint(bit);

	return irq;
}

#ifdef FUSB_USE_USB_ID
static int fusb_configure_id_intr(struct fusb_notify_control *fusb)
{
	int irq;

	irq = fusb_configure_gpio(FUSB_ID_GPIO_PORT, FUSB_ID_GPIO_BIT, fusb_interrupt_usb_id, FUSB_ID_INTR_NAME);
	if (0 > irq)
		return -1;

	fusb->usb_id_irqn = (unsigned int)irq;
	return 0;
}
#endif

static int fusb_configure_vbus_intr(struct fusb_notify_control *fusb)
{
	int irq;

	irq = fusb_configure_gpio(FUSB_VBUS_GPIO_PORT, FUSB_VBUS0_GPIO_BIT, fusb_interrupt_usb_vbus0, FUSB_VBUS0_INTR_NAME);
	if (0 > irq)
		return -1;

	fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT0] = (unsigned int)irq;

	irq = fusb_configure_gpio(FUSB_VBUS_GPIO_PORT, FUSB_VBUS1_GPIO_BIT, fusb_interrupt_usb_vbus1, FUSB_VBUS1_INTR_NAME);
	if (0 > irq)
		return -1;

	fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT1] = (unsigned int)irq;
	return 0;
}

static UDIF_ERR fusb_notify_otg_probe(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	switch(ch) {
	case FUSB_OTG_REG_CH:
		fusb->otg_regs = udif_devio_virt(dev, ch);
		fusb_set_gpio_base();
		fusb_configure_vbus_intr(fusb);
#ifdef FUSB_USE_USB_ID
		fusb_configure_id_intr(fusb);
#endif
		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		fusb_print_err("Out of setting channel");
		return UDIF_ERR_PAR;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_remove(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	switch(ch) {
	case FUSB_OTG_REG_CH:
#ifdef FUSB_USE_USB_ID
		free_irq(fusb->usb_id_irqn, NULL);
#endif
		free_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT0], NULL);
		free_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT1], NULL);
		fusb->otg_regs = NULL;
		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		fusb_print_err("Out of setting channel");
		return UDIF_ERR_PAR;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_suspend(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return UDIF_ERR_NOENT;
	}

	fusb->init_comp_flag = FUSB_OTG_FLAG_OFF;

	if (fusb->start_comp_flag == FUSB_OTG_FLAG_OFF) {
		return UDIF_ERR_OK;
	}

	switch(ch) {
	case FUSB_OTG_REG_CH:
		disable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT0]);
		disable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT1]);
#ifdef FUSB_USE_USB_ID
		disable_irq(fusb->usb_id_irqn);
#endif
		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		break;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_resume(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	int cid;
	enum usb_otg_vbus_stat vbus[USB_OTG_VBUS_PORT_MAX];

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return UDIF_ERR_NOENT;
	}

	fusb_setup_gpioint(FUSB_VBUS0_GPIO_BIT);
	fusb_setup_gpioint(FUSB_VBUS1_GPIO_BIT);
#ifdef FUSB_USE_USB_ID
	fusb_setup_gpioint(FUSB_ID_GPIO_BIT);
#endif

	if (fusb->start_comp_flag == FUSB_OTG_FLAG_OFF)
		return UDIF_ERR_OK;

	switch(ch) {
	case FUSB_OTG_REG_CH:
		enable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT0]);
		enable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT1]);
		vbus[USB_OTG_VBUS_PORT0] = fusb_read_vbus0();
		vbus[USB_OTG_VBUS_PORT1] = fusb_read_vbus1();
		fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT0], USB_OTG_VBUS_PORT0, false);
		fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT1], USB_OTG_VBUS_PORT1, false);

#ifdef FUSB_USE_USB_ID
		enable_irq(fusb->usb_id_irqn);
#endif

		if (fusb_read_id()) {
			cid = FUSB_B_DEVICE;
		} else {
			cid = FUSB_A_DEVICE;
		}
		fusb_notify_cid(fusb, cid, false);

		break;
	case FUSB_USB_VBUS_CH:
		break;
	case FUSB_USB_ID_CH:
		break;
	default :
		break;
	}

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static UDIF_ERR fusb_notify_otg_shutdown(const UDIF_DEVICE * dev, UDIF_CH ch, UDIF_VP data)
{
	struct fusb_notify_control *fusb;
	UDIF_ERR ret;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb_create_proc_flag == FUSB_OTG_FLAG_ON) {
		if ((ret = udif_remove_proc(&fusb_proc)) != UDIF_ERR_OK) {
			fusb_print_err("udif_remove_proc is failed");
		} else {
			fusb_create_proc_flag = FUSB_OTG_FLAG_OFF;
		}
	}

	fusb->start_comp_flag = FUSB_OTG_FLAG_OFF;

	udif_tasklet_kill(&fusb->event_tasklet);
	udif_kfree((UDIF_VP)fusb);

	fusb_func_trace("END");

	return UDIF_ERR_OK;
}

static void fusb_signal_event(UDIF_ULONG data)
{
	struct fusb_notify_control *fusb;

	UDIF_LIST temp_anchor;
	struct fusb_otg_event_container *ec;
	struct fusb_otg_event_container *temp_ec;

	if (data == 0) {
		return;
	}

	fusb = (struct fusb_notify_control *)data;

	udif_list_head_init(&temp_anchor);
	fusb_lock_irq(&fusb->lock, fusb->flags);
	udif_list_for_each_entry_safe(ec, temp_ec, &fusb->event_anchor, event_list) {
		udif_list_del(&ec->event_list);
		udif_list_head_init(&ec->event_list);
		udif_list_add_tail(&ec->event_list, &temp_anchor);
	}
	fusb_unlock_irq(&fusb->lock, fusb->flags);

	udif_list_for_each_entry_safe(ec, temp_ec, &temp_anchor, event_list) {
		udif_list_del(&ec->event_list);

		if (fusb != NULL) {
			fusb_print_dbg("usb_otg_core_notify. event type = %d \n",ec->event.type);
			usb_otg_core_notify(fusb->notify, &ec->event);
		}
		udif_kfree(ec);
	}

	return;
}

static void fusb_signal_event_interrupt(struct fusb_notify_control *fusb, union usb_otg_event *event)
{
	struct fusb_otg_event_container *event_container;

	event_container = (struct fusb_otg_event_container *)udif_kmalloc("fusb_usb_event", sizeof(struct fusb_otg_event_container), 0);
	if (event_container == NULL) {
		fusb_print_err("udif_kmalloc is failed");
		return;
	}

	memcpy(&event_container->event, event, sizeof(union usb_otg_event));
	udif_list_head_init(&event_container->event_list);
	fusb_lock_irq(&fusb->lock, fusb->flags);
	udif_list_add_tail(&event_container->event_list, &fusb->event_anchor);
	fusb_unlock_irq(&fusb->lock, fusb->flags);

	udif_tasklet_schedule(&fusb->event_tasklet);

	return;
}

#ifdef FUSB_USE_USB_ID
static irqreturn_t fusb_interrupt_usb_id(int irq, void *p)
{
	struct fusb_notify_control *fusb;
	int cid;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;

	if (fusb_read_id()) {
		cid = FUSB_B_DEVICE;
	} else {
		cid = FUSB_A_DEVICE;
	}

	fusb_notify_cid(fusb, cid, false);

	fusb_func_trace("END");

	return IRQ_HANDLED;
}
#endif

static irqreturn_t fusb_interrupt_usb_vbus0(int irq, void *p)
{
	struct fusb_notify_control *fusb;
	int vbus;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;

	vbus  = fusb_read_vbus0();
	fusb_notify_vbus(fusb, vbus, USB_OTG_VBUS_PORT0, false);

	fusb_func_trace("END");

	return IRQ_HANDLED;
}

static irqreturn_t fusb_interrupt_usb_vbus1(int irq, void *p)
{
	struct fusb_notify_control *fusb;
	int vbus;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;

	vbus  = fusb_read_vbus1();
	fusb_notify_vbus(fusb, vbus, USB_OTG_VBUS_PORT1, false);

	fusb_func_trace("END");

	return IRQ_HANDLED;
}

static void fusb_notify_cid(struct fusb_notify_control *fusb, int current_cid, bool forced)
{
	union usb_otg_event otg_event;

	if ((fusb->cid != current_cid) || forced) {
		otg_event.type              = USB_OTG_EVENT_TYPE_CID;
		otg_event.cid.value         = current_cid;
		otg_event.cid.port          = 0;
		fusb_signal_event_interrupt(fusb, &otg_event);
		fusb->cid = current_cid;
		fusb_print_dbg("current_cid = %d \n",current_cid);
	}
}

static void fusb_notify_vbus(struct fusb_notify_control *fusb, enum usb_otg_vbus_stat current_vbus, enum usb_otg_port_type port, bool forced)
{
	union usb_otg_event otg_event;

	if ((fusb->vbus[port] != current_vbus) || forced) {
		otg_event.type              = USB_OTG_EVENT_TYPE_VBUS;
		if (current_vbus == FUSB_VBUS_ON) {
			otg_event.vbus.vbus_stat = USB_OTG_VBUS_STAT_VALID;
		} else {
			otg_event.vbus.vbus_stat = USB_OTG_VBUS_STAT_OFF;
		}
		otg_event.vbus.port         = port;
		fusb_signal_event_interrupt(fusb, &otg_event);
		fusb->vbus[port] = current_vbus;
		fusb_print_dbg("current_vbus = %d \n",current_vbus);
		if( fusb->vbus[port] == FUSB_VBUS_OFF ){
			fusb->gadget_con = FUSB_EVENT_DISCONNECT;
		}
	}
}

int fusb_try_start_notify(void)
{
	struct fusb_notify_control *fusb;
	int cid;
	enum usb_otg_vbus_stat vbus[USB_OTG_VBUS_PORT_MAX];

	fusb = g_fusb_notify_control;
	fusb_assert_NULL(fusb != NULL);

#ifdef FUSB_USE_USB_ID
	enable_irq(fusb->usb_id_irqn);
#endif
	enable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT0]);
	enable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT1]);

	fusb->start_comp_flag = FUSB_OTG_FLAG_ON;

	if (fusb_read_id()) {
		cid = FUSB_B_DEVICE;
	} else {
		cid = FUSB_A_DEVICE;
	}
	vbus[USB_OTG_VBUS_PORT0]    = fusb_read_vbus0();
	vbus[USB_OTG_VBUS_PORT1]    = fusb_read_vbus1();

	fusb_notify_cid(fusb, cid, true);
	fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT0], USB_OTG_VBUS_PORT0, true);
	fusb_notify_vbus(fusb, vbus[USB_OTG_VBUS_PORT1], USB_OTG_VBUS_PORT1, true);

/*	current_otg_mode = USB_OTG_CONTROL_IDLE;*/

	return 0;
}
#ifdef CONFIG_USB_FUSB_OTG_TEST
EXPORT_SYMBOL_GPL(fusb_try_start_notify);
#endif

int fusb_try_stop_notify(void)
{
	struct fusb_notify_control *fusb;

	fusb = g_fusb_notify_control;
	fusb_assert_NULL(fusb != NULL);

	fusb->start_comp_flag = FUSB_OTG_FLAG_OFF;

#ifdef FUSB_USE_USB_ID
	disable_irq(fusb->usb_id_irqn);
#endif
	disable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT0]);
	disable_irq(fusb->usb_vbus_irqn[USB_OTG_VBUS_PORT1]);

	udif_tasklet_kill(&fusb->event_tasklet);

/*	current_otg_mode = USB_OTG_CONTROL_STOP;*/

	return FUSB_OTG_OK;
}
#ifdef CONFIG_USB_FUSB_OTG_TEST
EXPORT_SYMBOL_GPL(fusb_try_stop_notify);
#endif

int usb_otg_control_register_core_notify(int (*notify) (void * p))
{
	struct fusb_notify_control *fusb;

	fusb_lock(&fusb_lock);

	fusb_func_trace("START");

	if (notify == NULL) {
		fusb_print_err("argument notify callback is set NULL");
		fusb_unlock(&fusb_lock);
		return -EINVAL;
	}

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		fusb_unlock(&fusb_lock);
		return -ENOENT;
	}

	fusb->notify = notify;

	fusb_unlock(&fusb_lock);

	fusb_func_trace("END");

	return 0;
}
EXPORT_SYMBOL(usb_otg_control_register_core_notify);         /**<  */

int usb_otg_control_unregister_core_notify(int (*notify) (void * p))
{
	struct fusb_notify_control *fusb;

	fusb_func_trace("START");

	if (notify == NULL) {
		fusb_print_err("argument notify callback is set NULL");
		return -EINVAL;
	}

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb_lock(&fusb_lock);

	if (fusb->notify != notify) {
		fusb_print_err("fusb->notify not match");
		fusb_unlock(&fusb_lock);
		return -EINVAL;
	}
	fusb->notify = NULL;

	fusb_unlock(&fusb_lock);

	fusb_func_trace("END");

	return 0;
}
EXPORT_SYMBOL(usb_otg_control_unregister_core_notify);       /**<  */

enum usb_otg_vbus_stat fusb_otg_get_vbus(void)
{
	struct fusb_notify_control *fusb;
	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return USB_OTG_VBUS_STAT_OFF;
	}

	if (fusb->vbus[USB_OTG_VBUS_PORT0] == FUSB_VBUS_ON) { // tentative.
		return USB_OTG_VBUS_STAT_VALID;
	}

	return USB_OTG_VBUS_STAT_OFF;
}
EXPORT_SYMBOL(fusb_otg_get_vbus);                     /**<  */

int fusb_otg_notify_vbuserror(void)
{
	struct fusb_notify_control *fusb;
	union usb_otg_event otg_event;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	otg_event.type = USB_OTG_EVENT_TYPE_VBUSERROR;
	fusb_signal_event_interrupt(fusb, &otg_event);

	fusb_func_trace("END");

	return FUSB_OTG_OK;
}
EXPORT_SYMBOL(fusb_otg_notify_vbuserror);             /**<  */

static void fusb_con_timer_func(unsigned long data)
{
	struct fusb_notify_control *fusb;
	union usb_otg_event otg_event;

	fusb_func_trace("START");

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return;
	}

	if ((fusb->gadget_con != FUSB_EVENT_CONNECT) && (data == FUSB_EVENT_CONNECT)) {
		fusb->gadget_con = FUSB_EVENT_CONNECT;
		otg_event.type = USB_OTG_EVENT_TYPE_CON;
		otg_event.con.value = 1U;
		fusb_signal_event_interrupt(fusb, &otg_event);
	}

	fusb_func_trace("END");

	return;
}

int fusb_otg_notify_connect(unsigned int con)
{
	struct fusb_notify_control *fusb;
	union usb_otg_event otg_event;
	unsigned long expires;

	if ((con != FUSB_EVENT_DISCONNECT) && (con != FUSB_EVENT_CONNECT)) {
		fusb_print_err("Out of setting argument");
		return -EINVAL;
	}

	fusb = g_fusb_notify_control;
	if (fusb == NULL) {
		fusb_print_err("OTG Structure NULL pointer");
		return -ENOENT;
	}

	fusb->con_timer.data = con;
	if (fusb->gadget_con != con) {
		if (con == FUSB_EVENT_CONNECT) {
			expires = jiffies + msecs_to_jiffies(FUSB_DEVICE_CONNETCT_TIMER);
			mod_timer(&fusb->con_timer, expires);
		} else {
			fusb->gadget_con = con;
			otg_event.type = USB_OTG_EVENT_TYPE_CON;
			otg_event.con.value = 0;
			fusb_signal_event_interrupt(fusb, &otg_event);
		}
	}

	return FUSB_OTG_OK;
}
EXPORT_SYMBOL(fusb_otg_notify_connect);               /**<  */

MODULE_LICENSE ("GPL"); 				/**<  */
