/*
 * Copyright 2005,2006,2010,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/moduleparam.h>
#include <linux/wait.h>
#include <linux/sched.h>

#include <asm/memory.h>
#include <linux/dma-mapping.h>

#include <asm/uaccess.h>
#include <asm/errno.h>

#include <linux/version.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 21)
#include <linux/usb/ch9.h>
#else
#include <linux/usb_ch9.h>
#endif

#include <linux/device.h>

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 29)
#include <linux/usb/gadget.h>
#else
#include <linux/usb_gadget.h>
#endif

#include <linux/byteorder/generic.h>

#ifdef CONFIG_OSAL_UDIF
#include <linux/udif/cdev.h>
#include <mach/udif/devno.h>
#include <linux/udif/cache.h>
#include <mach/noncache.h> /* VA_TO_NONCACHE Diadem Only*/
#else
#error OSAL_UDIF required
#endif

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>

#include <linux/usb/specific_gadget/usbg_sen.h>

#include <linux/usb/specific_gadget/usb_ioctl.h>

#include <linux/usb/specific_gadget/usb_kdbg.h>

//#define USBG_SEN_DEBUG_INF
//#define USBG_SEN_DEBUG_API_DETAIL

#define FNAME "usbg_sen_main.c"

#include "usbg_sen_ldisc.h"
#define __USBG_SEN_MAIN_PVT__
#include "usbg_sen_main.h"
#include "usbg_sen_conf.h"

#include "usbg_sen_fifo.h"

MODULE_AUTHOR("Sony Corporation");
MODULE_DESCRIPTION(USBG_SEN_DESC);
MODULE_VERSION(USBG_SEN_VER);
MODULE_LICENSE("GPL");

#if !defined(SENSER_KMALLOC_XFERBUF)
static unsigned char* sen_xfer_buf = NULL;
UDIF_MODULE_PARAM(sen_xfer_buf, charp, 0644);
static int sen_xfer_buf_size = 0; 
UDIF_MODULE_PARAM(sen_xfer_buf_size, int, 0644);
static int _setup_buf(void *arg);
#endif

/**
  *	file operations
 */
static struct UDIF_CDEV_OPS fops_se = {
	.open  = fops_se_open,
	.close = fops_se_release,
	.ioctl = fops_ioctl,
};

static struct UDIF_CDEV_OPS fops_te = {
	.open  = fops_te_open,
	.close = fops_te_release,
	.ioctl = fops_ioctl,
};

#ifdef CONFIG_OSAL_UDIF
static struct usbg_cdev sonymisc_se = {
    .node = udif_device_node(UDIF_NODE_USB_SEN),
    .fops = &fops_se,
};
static struct usbg_cdev sonymisc_te = {
    .node = udif_device_node(UDIF_NODE_USB_TER),
    .fops = &fops_te,
};
#else
#error OSAL_UDIF required
#endif

/**
  *	default data -
  */
const static struct usbg_se_user default_usbg_se_user = {
	NULL,
};

const static struct usbg_te_user default_usbg_te_user = {
	NULL,
};

/**
  *	xfer buffer-
  */
static unsigned char* P_SEN_TX_BUF = NULL;
static int s_TX_BUF_SIZE = 0; 
static unsigned char* P_SEN_RX_FIFO_BUF = NULL;


/* ΤȤȾü˥ǡĤ뤳ȤΤfifoŪʻȤ */
static struct sen_xfer_fifo  s_rx_fifo;

#define PERR(fmt, args...) {printk("%s(%d) "fmt, __FUNCTION__,__LINE__,##args); }

#ifdef USBG_SEN_SEMAPHORE
 #ifdef CONFIG_OSAL_UDIF
  static  UDIF_MUTEX  usbg_ep_se_lock;         // lock for SE endpoints
  static  UDIF_MUTEX  usbg_comp_usb_req_lock;  // lock for Completion of USB request
  
  #define USBG_SEN_LOCK_INIT()        { udif_mutex_init( &usbg_ep_se_lock        ); \
                                        udif_mutex_init( &usbg_comp_usb_req_lock ); }
  
  #define USBG_SEN_EP_SE_LOCK_ON()          { udif_mutex_lock(   &usbg_ep_se_lock ); \
                                              USBG_SEN_INF("### SE_ENDPOINT LOCK ON  ---> \n"); }
  
  #define USBG_SEN_EP_SE_LOCK_OFF()         { udif_mutex_unlock( &usbg_ep_se_lock ); \
                                              USBG_SEN_INF("### SE_ENDPOINT LOCK OFF <--- \n"); }
  
  #define USBG_SEN_COMP_USB_REQ_LOCK_ON()   { udif_mutex_lock(   &usbg_comp_usb_req_lock ); \
                                              USBG_SEN_INF("### COMP_USB_REQ LOCK ON  ---> \n"); }
  
  #define USBG_SEN_COMP_USB_REQ_LOCK_OFF()  { udif_mutex_unlock( &usbg_comp_usb_req_lock ); \
                                              USBG_SEN_INF("### COMP_USB_REQ LOCK OFF <--- \n"); }
  
 #else  // CONFIG_OSAL_UDIF
  #error
 #endif  // CONFIG_OSAL_UDIF
#else  // USBG_SEN_SEMAPHORE
 #error
#endif  // USBG_SEN_SEMAPHORE

/*
  *	instance -
  */

struct usbg_sen_ep_data* ptbl_ep_data[USBG_EPID_NBROF] = {0, };

/**
  *	fops_se_open	- senser node open
  */
static UDIF_ERR fops_se_open(UDIF_FILE *filp)
{
	struct file file;
	struct file *fd = &file;
	UDIF_ERR err = UDIF_ERR_OK;
	struct func_data* p_this;

	/* <framwork>	*/
	if( usbg_cmn_new( NULL, fd ) ){
		err = UDIF_ERR_DEV;
		KERR("%s() err=%d\n", __FUNCTION__, err);
		goto SUB_RET;
	}
	p_this = fd->private_data;
    udif_file_data(filp) = fd->private_data;
	/* </framework>	*/

	/* <user>	*/
	/* allocate private data	*/
	p_this->vp_user = kmalloc( sizeof(struct usbg_se_user),
				   GFP_ATOMIC );
	if ( p_this->vp_user == NULL ) {
		KERR("%s() p_this->vp_user == NULL\n", __FUNCTION__);
		usbg_cmn_delete( p_this );
		err = (UDIF_ERR)-ENOBUFS;
		goto SUB_RET;
	}

	/* initialize private data	*/
	memcpy(p_this->vp_user, &default_usbg_se_user,
	       sizeof(struct usbg_se_user) );
	/* </user>	*/

SUB_RET:
	return err;
}
static UDIF_ERR fops_te_open(UDIF_FILE *filp)
{
	struct file file;
	struct file *fd = &file;
    UDIF_ERR err = UDIF_ERR_OK;
	struct func_data* p_this;

	/* <framwork>	*/
	if( usbg_cmn_new( NULL, fd ) ){
		err = UDIF_ERR_DEV;
		KERR("%s() err=%d\n", __FUNCTION__, err);
		goto SUB_RET;
	}
	p_this = fd->private_data;
    udif_file_data(filp) = fd->private_data;
	/* </framework>	*/

	/* <user>	*/
	/* allocate private data	*/
	p_this->vp_user = kmalloc( sizeof(struct usbg_te_user),
				   GFP_ATOMIC );
	if ( p_this->vp_user == NULL ) {
		KERR("%s() p_this->vp_user == NULL\n", __FUNCTION__);
		usbg_cmn_delete( p_this );
		err = (UDIF_ERR)-ENOBUFS;
		goto SUB_RET;
	}

	/* initialize private data	*/
	memcpy(p_this->vp_user, &default_usbg_te_user,
	       sizeof(struct usbg_te_user) );
	
	/* </user>	*/
SUB_RET:
	return err;
}

/**
  *	fops_se_release	- senser node release
  */
static UDIF_ERR fops_se_release(UDIF_FILE *filp)
{
	int err = 0;
	struct func_data* p_this = (struct func_data*)udif_file_data(filp);
	
	if(p_this==NULL){
		err = (UDIF_ERR)-ENOBUFS;
		KERR("%s() err=%d\n", __FUNCTION__, err);
		goto SUB_RET;
	}
	
	if( p_this->vp_user ){
		kfree( p_this->vp_user );
		p_this->vp_user=NULL;
	}

	/* 󥹥󥹤β	*/
	usbg_cmn_delete( p_this );
	udif_file_data(filp) = NULL;

SUB_RET:
	return 0;
}

/**
  *	fops_te_release	- senser node release
  */
static UDIF_ERR fops_te_release(UDIF_FILE *filp)
{
	int err = 0;
	struct func_data* p_this = (struct func_data*)udif_file_data(filp);
	if(p_this==NULL){
		err = (UDIF_ERR)-ENOBUFS;
		KERR("%s() err=%d\n", __FUNCTION__, err);
		goto SUB_RET;
	}

	if( p_this->vp_user ){
		kfree( p_this->vp_user );
		p_this->vp_user=NULL;
	}

	/* 󥹥󥹤β	*/
	usbg_cmn_delete( p_this );
	udif_file_data(filp) = NULL;

SUB_RET:
	return 0;
}

/**
  *	fops_se_ioctl	-
  */
static UDIF_ERR fops_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
	struct file file;
	struct file *fd = &file;
	unsigned int cmd = param->cmd;
	unsigned long arg = param->arg;
	UDIF_ERR ret = 0;

    fd->private_data = udif_file_data(filp);
	if( _IOC_TYPE(cmd) == USB_IOC_CMN ){
		ret = fops_cmn_ioctl( NULL, fd, cmd, arg );
	} else if( _IOC_TYPE(cmd) == USB_IOC_SE ) {
		ret = fops_se_ioctl( NULL, fd, cmd, arg );
	} else {
		ret = UDIF_ERR_NOTTY;
		KERR("%s() invalid seq. number=%d, ret=%d\n",
		     __FUNCTION__, _IOC_TYPE(cmd), ret);
		goto SUB_RET;
	}
    udif_file_data(filp) = fd->private_data;

SUB_RET:
	return (long)ret;
}

/**
  * fops_se_ioctl_cmn	-
  */
  
static int fops_cmn_ioctl( struct inode *inode, struct file *fd,
			      unsigned int cmd, unsigned long arg)
{
	int err = 0;

	if( _IOC_NR(cmd) >= USB_IOC_CMN_NBROF ){
		err = -ENOTTY;
		KERR("%s() invalid ioctl seq. number=%d, ret=%d\n",
		     __FUNCTION__, _IOC_NR(cmd), err);
		goto SUB_RET;
	}

	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd) );
	} else if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd) );
	}
	if(err){
		err = -EFAULT;
		KERR("%s() invalid ioctl dir\n", __FUNCTION__);
		goto SUB_RET;
	}

	switch (cmd) {
	case USB_IOC_CMN_PROBE:
		err = fops_cmn_ioctl_probe(inode, fd, (void *) arg);
		break;
	case USB_IOC_CMN_REMOVE:
		err = fops_cmn_ioctl_remove(inode, fd, (void *) arg);
		break;
	default:
		err = -ENOTTY;
		break;
	}

	if(err){
		KERR("%s() err=%d\n", __FUNCTION__, err);
	}

SUB_RET:
	return err;
}
/**
  *	fops_se_ioctl_sub	-
  */
static int fops_se_ioctl(struct inode *inode, struct file *fd,
			 unsigned int cmd, unsigned long arg)
{
	int err = -ENOSYS;

	if( _IOC_NR(cmd) >= USBG_IOC_SE_NBROF ){
		err = -ENOTTY;
		KERR("%s() invalid ioctl seq. number=%d\n", __FUNCTION__, _IOC_NR(cmd));
		goto SUB_RET;
	}

	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd) );
	} else if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd) );
	}
	if(err){
		err = -EFAULT;
		KERR("%s() invalid ioctl dir\n", __FUNCTION__);
		goto SUB_RET;
	}

	switch(cmd){
	case USBG_IOC_SE_SEND :
		err = fops_se_ioctl_send(inode, fd, (void *) arg);
		break;
	case USBG_IOC_SE_RECEIVE :
		err = fops_se_ioctl_receive(inode, fd, (void *) arg);
		break;
	case USBG_IOC_SE_SETBUF :
#if !defined(SENSER_KMALLOC_XFERBUF)
		err = _setup_buf((void *)arg);
#endif
		break;
	default:
		err = -ENOTTY;
		break;
	}

	if(err){
		KERR("%s() invalid ioctl seq. err=%d\n", __FUNCTION__, err);
	}

SUB_RET:
	return err;
}

/**
  *	fops_cmn_ioctl_probe	-
  */
static int fops_cmn_ioctl_probe(struct inode *inode,
				struct file *fd, void *p_arg)
{
	int err = -ENOSYS;
	struct func_data* p_this = (struct func_data*)(fd->private_data);
	struct probe_info *p_info = p_arg;
	struct usbg_sen_probe_info* p_sen_probe=NULL;
	struct usbg_se_user* p_user = (struct usbg_se_user*)(p_this->vp_user);

	/* <framework>	*/
	/* copy probe param.	*/
	if( usbg_cmn_copy_probe_info(p_this, p_info) ){
		err = -ENOBUFS;
		KERR("%s() copy probe info\n", __FUNCTION__);
		goto SUB_RET;
	}
	if(p_this->info.uc_func_info_interface_num > USBG_SEN_IF_NR){
		err = -ENOSYS;
		KERR("%s() if num too big\n", __FUNCTION__);
		KERR("                 %d\n",
		     p_this->info.uc_func_info_interface_num);
		KERR("                 %d\n",USBG_SEN_IF_NR);
		goto SUB_RET;
	}
	KINFO("%s() if num=%d\n",
	      __FUNCTION__, p_this->info.uc_func_info_interface_num);

	p_this->core_reg_info.function = NULL;
	p_this->core_reg_info.context = p_this;

	p_this->core_reg_info.config
		= p_this->info.tbl_interface[0].uc_configuration;

	p_this->core_reg_info.interface
		= p_this->info.tbl_interface[0].uc_interface;

	p_this->core_reg_info.start = core_start;
	p_this->core_reg_info.stop = core_stop;
	p_this->core_reg_info.vendor = core_vendor;

	err = usb_gadgetcore_register_driver(&p_this->core_reg_info);

	/* </framework>	*/

	if(err){
		KERR("%s() err=%x\n", __FUNCTION__, err);
	} else {
		KINFO("%s() err=%x\n\n", __FUNCTION__, err);
	}

	/* <user>	*/
	p_sen_probe = (struct usbg_sen_probe_info*)p_this->info.vp_user;

	if(p_sen_probe){
		err = alloc_usbg_se_user(p_user);
		if(err==0){
			memcpy(p_user->p_cmd,
				       p_sen_probe->cmd,
				       sizeof(struct usbg_sen_cmd)
				       * USBG_SEN_CMD_NR);
		}
	}
	/* </user>	*/
SUB_RET:
	return err;
}

/**
  *	fops_cmn_ioctl_remove	-
  */
static int fops_cmn_ioctl_remove(struct inode *inode, struct file *fd,
				void *p_arg)
{
	struct func_data* p_this = (struct func_data*)(fd->private_data);
	struct usbg_se_user* p_user = (struct usbg_se_user*)(p_this->vp_user);
	cleanup_usbg_se_user(p_user);
	
	/* <framework>	*/
	/* usb_gadget_core_unregister		*/
	usb_gadgetcore_unregister_driver(&p_this->core_reg_info);

	/* </frameowrk>	*/
	/* <framework>	*/
	/* probe infoꥢ	*/
	usbg_cmn_clear_probe_info( p_this );
	/* </frameowrk>	*/
	return 0;
}

/**
  *	fops_se_ioctl_send	-	send senser data
  */

static int fops_se_ioctl_send(struct inode *inode, struct file *fd, void *p_arg)
{
    int err = 0;
    ssize_t res = 0;

    struct usbg_se_data user_req;
    struct func_data* p_this = (struct func_data*)(fd->private_data);
    struct usbg_se_data* p_param = (struct usbg_se_data*)p_arg;
    struct usb_ep* ep =NULL;
    struct usbg_sen_ep_data* ep_data=NULL;
    struct usbg_se_ep_data* ep_data_se=NULL;

    ssize_t done = 0;
    int req_remain; /* Ĥ */
    uint32_t mode; /* user׵ᤷž⡼ */
    unsigned char* p_user_buf; /* žХåե */
    int current_transfer_size = s_TX_BUF_SIZE; /* žХåե˽ */

    USBG_SEN_API_DETAIL(" Entry >>>\n");

    /* parameter copy	*/
    if(copy_from_user(&user_req, p_param,
                      sizeof(struct usbg_se_data))){
        err = -EBUSY;
        PERR("\n");
        goto SUB_RET;
    }

    /* copy parameter */
    req_remain = user_req.count;
    mode = user_req.mode;

    if( mode & USBG_SEN_XFERMODE_PHYS ){
        p_user_buf = (unsigned char*)PHYS_TO_VA((unsigned long)user_req.cp);
    }else{
        p_user_buf = user_req.cp;
    }

#ifdef USBG_SEN_DEBUG_API_DETAIL
    if( mode & USBG_SEN_XFERMODE_PHYS ){
        USBG_SEN_API_DETAIL("Phys Buffer Mode req_size = %d phys_addr = %p\n", req_remain, user_req.cp);
    }else{
        USBG_SEN_API_DETAIL("Norm Buffer Mode req_size = %d user_addr = %p\n", req_remain, user_req.cp);
    }
#endif

    /* ޤ */
    while( req_remain > 0 ){
        current_transfer_size = s_TX_BUF_SIZE;
        if( current_transfer_size > req_remain ){
            /* žХåե׵᤬ʤФˤ碌 */
            current_transfer_size = req_remain;
        }

        /* žХåեǡ򥳥ԡ */
        if( mode & USBG_SEN_XFERMODE_PHYS){
            memcpy(P_SEN_TX_BUF, p_user_buf, current_transfer_size);
        }else{
            if(copy_from_user(P_SEN_TX_BUF, p_user_buf, current_transfer_size)){
                err = -EBUSY;
                PERR("\n");
                goto SUB_RET;
            }
        }

        USBG_SEN_EP_SE_LOCK_ON();
        do {	/* while(0)	*/
            /* get ep info	*/
            ep = USBG_SEN_GET_EP(p_this, USBG_EPID_SE_TX);
            if(ep==NULL){
                err = 0;
                res = -EAGAIN;
                PERR("\n");
                break;
            }

            ep_data = ptbl_ep_data[USBG_EPID_SE_TX];
            if(ep_data==NULL){
                err = 0;
                res = -EAGAIN;
                PERR("\n");
                break;
            }

            ep_data_se = (struct usbg_se_ep_data*)ep_data->pvt;
            ep_data_se->user_req= &user_req;

            if(test_and_set_bit( USBG_SE_FLAG_USING,
                                 &ep_data->wait_flag )){
                err = 0;
                res= -EBUSY;
                PERR("\n");
                break;
            }

            set_bit(USBG_SE_FLAG_XFER_REQ, &ep_data->wait_flag);
            /* create request */
            ep_data->usb_req->buf = P_SEN_TX_BUF;
            if( mode &  USBG_SEN_XFERMODE_DMA){
#ifdef CONFIG_ARCH_CXD900XX /* tentative */
                ep_data->usb_req->dma
                    = (dma_addr_t)VA_TO_PHYS((unsigned long)ep_data->usb_req->buf);
#else
                ep_data->usb_req->dma
                    = (dma_addr_t)__virt_to_bus((unsigned long)ep_data->usb_req->buf);
#endif
            }else
            {
                ep_data->usb_req->dma
                    = (dma_addr_t)NULL;
            }
            ep_data->usb_req->length = current_transfer_size;
            /*	ep_data->usb_req->no_interrupt = 0;	*/
            if( current_transfer_size == req_remain ){
                /* ǺǸʤzeroĤ */
                ep_data->usb_req->zero = 1;
            }else{
                ep_data->usb_req->zero = 0;
            }
            ep_data->usb_req->context
                = (void*)p_this;
            ep_data->usb_req->complete
                = comp_se_ioctl_send;

            clear_bit( USBG_SE_FLAG_XFER_INTR, &ep_data->wait_flag );

            if(mode & USBG_SEN_XFERMODE_DMA){
#ifdef CONFIG_OSAL_UDIF
                udif_cache_ctrl(ep_data->usb_req->buf,
                                ep_data->usb_req->length, UDIF_CACHE_FLUSH);
#else
#error OSAL_UDIF required
#endif
            }

            if(test_bit(USBG_SE_FLAG_XFER_REQ,&ep_data->wait_flag)){
                USBG_SEN_API_DETAIL("usb_ep_queue( %p, %p ) size = %d\n", ep, ep_data->usb_req, ep_data->usb_req->length);
                err = usb_ep_queue( ep, ep_data->usb_req, GFP_ATOMIC );
            } else {
                err = 0;
                res = -EAGAIN;
                PERR("\n");
            }
        } while(0);
        USBG_SEN_EP_SE_LOCK_OFF();

        switch(err){
          case 0:
            break;
          case -ESHUTDOWN:
            err = 0;
            res = -EAGAIN;
            PERR("\n");
            break;
          case -EINVAL:
          default:
            err = -ENOSYS;
            PERR("\n");
            break;
        }
        if(err || res){
            goto SUB_RET;
        }

        if(wait_event_interruptible(
            ep_data->wait_que,
            test_and_clear_bit(USBG_SE_FLAG_XFER_COMP,
                               &ep_data->wait_flag))){
            KERR("%s() interrupted\n", __FUNCTION__);
            res = -EINTR;
            PERR("\n");
            goto SUB_RET;
        }

        if(test_bit( USBG_SE_FLAG_XFER_INTR, &ep_data->wait_flag )){
            res = -EINTR;
            PERR("\n");
            goto SUB_RET;
        }

        USBG_SEN_COMP_USB_REQ_LOCK_ON();
        switch(ep_data->comp_usb_req.status){
          case 0:
            done += ep_data->comp_usb_req.actual;
            /* Τǥ󥿤 */
            req_remain -= ep_data->comp_usb_req.actual;
            p_user_buf += ep_data->comp_usb_req.actual;
            break;
          case -ESHUTDOWN:
            res = -EINTR;
            PERR("\n");
            break;
          default:
            err = -ENOSYS;
            PERR("\n");
        }
        USBG_SEN_COMP_USB_REQ_LOCK_OFF();

        clear_bit(USBG_SE_FLAG_USING, &ep_data->wait_flag);
        if( err != 0 || res < 0 ){
            PERR("error\n");
            goto SUB_RET; // Errorʤ齪λ
        }
    } /* while ( req_remain > 0 ) */

    /* OK */
    res = done;
    
 SUB_RET:
    if(copy_to_user(&((struct usbg_se_data*)p_arg)->count, &res, sizeof(res))){
        PERR("\n");
    }
    USBG_SEN_API_DETAIL("<<<< EXIT err = %d, res = %d\n", err, res);
    return err;
}

static void comp_se_ioctl_send(struct usb_ep *ep, struct usb_request *req)
{
	struct usbg_sen_ep_data* ep_data;
	
	USBG_SEN_COMP_USB_REQ_LOCK_ON();
	ep_data = ptbl_ep_data[USBG_EPID_SE_TX];
	memcpy(&ep_data->comp_usb_req,req, sizeof(struct usb_request));
	USBG_SEN_COMP_USB_REQ_LOCK_OFF();
	
	set_bit(USBG_SE_FLAG_XFER_COMP, &ep_data->wait_flag);
	wake_up_interruptible(&ep_data->wait_que);
}

/**
  *	fops_se_ioctl_receive -	
  */
static int fops_se_ioctl_receive(struct inode *inode,
				 struct file *fd, void *p_arg)
{
	int err = 0;
	ssize_t res = 0;

	struct usbg_se_data user_req;
	struct func_data* p_this = (struct func_data*)(fd->private_data);
	struct usbg_se_data* p_param = (struct usbg_se_data*)p_arg;
	struct usb_ep *ep=NULL;
	struct usbg_sen_ep_data *ep_data=NULL;
	struct usbg_se_ep_data* ep_data_se=NULL;

    /* MaxPθEOVERFLOWʤ褦ѹ */
    unsigned  maxp; /* Max Packet Size */
    int req_remain; /* ̤Ϥɬפǡ */
    uint32_t mode; /* userȤ׵ᤷž⡼ */
    unsigned char* p_user_buf; /* žХåե */
    ssize_t done = 0;

    USBG_SEN_INF("ENTRY\n");

    /* parameter copy	*/
    if(copy_from_user(&user_req, p_param, sizeof(struct usbg_se_data))){
        err = -EBUSY;
        PERR("\n");
        goto SUB_RET;
    }
    mode = user_req.mode;
    req_remain = user_req.count;
    if( mode & USBG_SEN_XFERMODE_PHYS ){
        p_user_buf = (unsigned char*)PHYS_TO_VA((unsigned long)user_req.cp);
        /* ǤCache AccessˤƤžХåեmemcpyȤFLUSH */
    }else{
        p_user_buf = user_req.cp;
    }

#ifdef USBG_SEN_DEBUG_API_DETAIL
    if( mode & USBG_SEN_XFERMODE_PHYS ){
        USBG_SEN_API_DETAIL("Phys Buffer Mode req_size = %d addr = %p\n", user_req.count, user_req.cp);
    }else{
        USBG_SEN_API_DETAIL("Norm Buffer Mode req_size = %d addr = %p\n", user_req.count, user_req.cp);
    }
#endif

    /* ̤׵ʬΥǡϤޤ */
    while( req_remain > 0 ){
        USBG_SEN_INF("while ( req_remain(%d) )\n", req_remain);


        /* 1. rx_fifo˥ǡ硢桼Ϥ */
        if( 0 != sen_fifo_data_size(&s_rx_fifo)){
            int copy_size;
            int count = 0;

            copy_size = sen_fifo_data_size(&s_rx_fifo);
            USBG_SEN_INF("%d bytes data in rx_fifo\n", copy_size );
            if( copy_size > req_remain ){
                /* fifoˤǡʬɬפʬ */
                copy_size = req_remain;
            }
            /* Copy FIFO to User Buffer */
            if( mode & USBG_SEN_XFERMODE_PHYS ){
                /* ʪɥ쥹ξ */
                count = sen_fifo_read(&s_rx_fifo, p_user_buf, copy_size );
#ifdef USBG_SEN_DEBUG_API_DETAIL                
                USBG_SEN_API_DETAIL("Dump physAddr(%p) 16Byte "
                                    "%02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X %02X\n",
                                    p_user_buf, p_user_buf[0],p_user_buf[1],p_user_buf[2],p_user_buf[3],p_user_buf[4],p_user_buf[5],p_user_buf[6],p_user_buf[7],
                                    p_user_buf[8],p_user_buf[9],p_user_buf[10],p_user_buf[11],p_user_buf[12],p_user_buf[13],p_user_buf[14],p_user_buf[15]);
#endif
            }else{
                /* User֥Хåեξ */
                count = sen_fifo_read_to_user(&s_rx_fifo, p_user_buf, copy_size );
            }
            req_remain -= count;
            p_user_buf += count;
            USBG_SEN_API_DETAIL("Next dest addr = %p\n", p_user_buf);
            done += count;
            /* while loopƬ */
            continue;
        }

        /* 2. FIFO˥ǡʤ */
        /* FIFO˥ǡ̵ȤˤϤʤ*/
        /* DMAΥɥ쥹FIFOƬ˸ꤹ뤿ᡢFIFO_RESET롣 */
        sen_fifo_reset(&s_rx_fifo);
        USBG_SEN_INF(" no data in rx_fifo done = %d, req_remain = %d\n"
                     , done, req_remain);
        USBG_SEN_EP_SE_LOCK_ON();

        do {	/* while(0)	*/

            int current_transfer_size = req_remain; /* DMAž륵 */
            
            /* get ep info	*/
            ep = USBG_SEN_GET_EP(p_this, USBG_EPID_SE_RX);
            if(ep==NULL){
                err = 0;
                res = -EAGAIN;
                PERR("\n");
                break;
            }

            /* maxp */
            maxp = ep->maxpacket;
            USBG_SEN_INF(" maxp = %d\n", maxp);

            ep_data = ptbl_ep_data[USBG_EPID_SE_RX];
            if(ep_data==NULL){
                err = 0;
                res = -EAGAIN;
                PERR("\n");
                break;
            }

            ep_data_se = (struct usbg_se_ep_data*)ep_data->pvt;
            ep_data_se->user_req= &user_req;

            if(test_and_set_bit( USBG_SE_FLAG_USING,
                                 &ep_data->wait_flag )){
                err = -EBUSY;
                PERR("");
                break;
            }

            set_bit(USBG_SE_FLAG_XFER_REQ, &ep_data->wait_flag);
            /* create usb_request	*/
            ep_data->usb_req->buf = sen_fifo_buf_top(&s_rx_fifo);
            if( mode & USBG_SEN_XFERMODE_DMA ){
#ifdef CONFIG_ARCH_CXD900XX /* tentative */
                ep_data->usb_req->dma
                    = (dma_addr_t)VA_TO_PHYS((unsigned long)ep_data->usb_req->buf);
#else
                ep_data->usb_req->dma
                    = (dma_addr_t)__virt_to_bus((unsigned long)ep_data->usb_req->buf);
#endif
            } else {
                ep_data->usb_req->dma
                    = (dma_addr_t)NULL;
            }

            /* žХåեžˤ */
            if( current_transfer_size > sen_fifo_free_size(&s_rx_fifo) ){
                current_transfer_size = sen_fifo_free_size(&s_rx_fifo);
            }

            /* maxp ñ̤ڤ夲reqeust */
            /* ̤12Byte, 4Byteȡڤ׵
               Hostϸ³ΥڥɤƱPacketϤᡢ̤׵Ĺ˽ȡ
               EOVERFLOWˤʤ׵ĹꡢHostϤǡ¿Τǡ */
            /* ݤƤžХåեΤmaxpñ̤ݾڤΤǡñڤ夲Ƥ
               ǥХåեСե뤳ȤϤʤ */
            current_transfer_size = ((current_transfer_size + (maxp-1))/maxp )* maxp;
            ep_data->usb_req->length = usb_gadgetcore_ep_align_maybe(ep, current_transfer_size);
            KINFO("%s() size=%d, aligned=%d\n", __FUNCTION__, current_transfer_size, ep_data->usb_req->length);
            //		ep_data->usb_req->zero = 1;
            ep_data->usb_req->zero = 0; //1ˤƤޤȡEOVERFLOWˤʤ롣0ԤƤΤˡʤѥåȤϤΤǡ//
            ep_data->usb_req->short_not_ok = 0;
            ep_data->usb_req->context
                = (void*)p_this;
            ep_data->usb_req->complete
                = comp_se_ioctl_receive;

            clear_bit( USBG_SE_FLAG_XFER_INTR, &ep_data->wait_flag );

            if(mode & USBG_SEN_XFERMODE_DMA){
                udif_cache_ctrl(ep_data->usb_req->buf,
                                ep_data->usb_req->length,
                                UDIF_CACHE_INVALIDATE);
            }

            if(test_bit(USBG_SE_FLAG_XFER_REQ,&ep_data->wait_flag)){
                USBG_SEN_API_DETAIL("usb_ep_queue( %p, %p ) size = %d\n", ep, ep_data->usb_req, ep_data->usb_req->length);
                err = usb_ep_queue( ep, ep_data->usb_req, GFP_ATOMIC );
            } else {
                err = 0;
                res = -EAGAIN;
                PERR("\n");
            }
        } while(0);
        USBG_SEN_EP_SE_LOCK_OFF();

        switch(err){
          case 0:
            break;
          case -ESHUTDOWN:
            err = 0;
            res = -EAGAIN;
            PERR("\n");
            break;
          case -EINVAL:
          default:
            err = -ENOSYS;
            PERR("\n");
            break;
        }
        if(err || res ){
            goto SUB_RET;
        }

        if(wait_event_interruptible(
            ep_data->wait_que,
            test_and_clear_bit(USBG_SE_FLAG_XFER_COMP,
                               &ep_data->wait_flag))){
            KERR("%s() interrupted\n", __FUNCTION__);
            res = -EINTR;
            PERR("\n");
            goto SUB_RET;
        }

        if(test_bit( USBG_SE_FLAG_XFER_INTR, &ep_data->wait_flag )){
            res = -EINTR;
            PERR("\n");
            USBG_SEN_API_DETAIL("-EINTR\n");
            goto SUB_RET;
        }

        USBG_SEN_COMP_USB_REQ_LOCK_ON();
        switch(ep_data->comp_usb_req.status){
          case 0:
            /* usb_requestOKcomplete */
            /* FIFOFillΤǡwp */
            USBG_SEN_INF("skip_wp actual = %d\n",ep_data->comp_usb_req.actual);
            sen_fifo_skip_wp( &s_rx_fifo, ep_data->comp_usb_req.actual );
            
            break;
          case -EOVERFLOW:
            res = -EOVERFLOW;
            PERR("EOVERFLOW ......\n");
            break;
          case -ESHUTDOWN:
            USBG_SEN_API_DETAIL("-ESHUTDOWN\n");
            res = -EINTR;
            PERR("\n");
            break;
          default :
            USBG_SEN_API_DETAIL("-ENOSYS\n");
            err = -ENOSYS;
            PERR("\n");
            break;
        }
        USBG_SEN_COMP_USB_REQ_LOCK_OFF();

        clear_bit(USBG_SE_FLAG_USING, &ep_data->wait_flag);
        if( err != 0 || res < 0 ){
            PERR("error\n");
            goto SUB_RET; // Errorʤ齪λ
        }

    } /* while( req_remain > 0 ); */
    USBG_SEN_INF("Escape from while( req_remain(%d) > 0 )\n", req_remain);

    /* OKξ */
    res = done;

    /* ʬΥǡCache Flush */
    if( mode & USBG_SEN_XFERMODE_PHYS ){
        udif_cache_ctrl((unsigned char*)PHYS_TO_VA((unsigned long)user_req.cp),
                        done, UDIF_CACHE_FLUSH);
    }

SUB_RET:
	if(copy_to_user(&p_param->count, &res, sizeof(res))){
		PERR("\n");
	}
    
    if( err != 0 || res < 0 ){
        /* Errorʤfiforead pointerreset */
        sen_fifo_reset(&s_rx_fifo);
    }
    USBG_SEN_API_DETAIL("<<<< EXIT err = %d, res = %d\n", err, res);
	return err;
}

static void comp_se_ioctl_receive(struct usb_ep *ep, struct usb_request *req)
{
	struct usbg_sen_ep_data *ep_data;

  USBG_SEN_COMP_USB_REQ_LOCK_ON();
	ep_data = ptbl_ep_data[USBG_EPID_SE_RX];
	memcpy(&ep_data->comp_usb_req,req, sizeof(struct usb_request));
  USBG_SEN_COMP_USB_REQ_LOCK_OFF();

   USBG_SEN_API_DETAIL(" ep_queue complete! req=%p, status = %d req->actual = %d\n", req, req->status, req->actual);
  
	set_bit(USBG_SE_FLAG_XFER_COMP, &ep_data->wait_flag);
	wake_up_interruptible(&ep_data->wait_que);
}

/*  */
/* core٥								*/
/*  */
/* start								*/
/*  */
static void core_start( struct usb_gadget_func_driver* p_drv,
                        unsigned char alt,
                        struct usb_gadget_ep_list list )
{
	int err = -ENOSYS;
	struct func_data* p_this = (struct func_data*)(p_drv->context);
	struct usb_ep* p_ep = NULL;
	struct usbg_sen_ep_data* ep_data = NULL;
	struct usbg_te_ep_data* ep_data_te = NULL;
	int i = 0;
	
	USBG_SEN_EP_SE_LOCK_ON();
	
	/* <framework>	*/
	if( usbg_cmn_make_ep_list( p_this, alt, list ) ){
		err = -ENOBUFS;
		KERR("%s() make ep list\n", __FUNCTION__);
		goto SUB_RET;
	}
	/* </framework>	*/

	/* <user>	*/
	for(i=0;i<list.num_ep;i++){
		p_ep = list.ep[i];
		ep_data = ptbl_ep_data[i];
		if(ep_data){
			memset(ep_data, 0,
			       sizeof(struct usbg_sen_ep_data));
		} else {
			KERR("%s() !ep_data\n", __FUNCTION__);
			err = -ENOBUFS;
			goto SUB_RET;
		}

		if(i==USBG_EPID_SE_TX
		   || i==USBG_EPID_TE_TX){
			ep_data->dir = USBG_SEN_DIR_TX;
		} else {
			ep_data->dir = USBG_SEN_DIR_RX;
		}
		
		if(i==USBG_EPID_SE_TX
		   || i==USBG_EPID_SE_RX){
			/* senser	*/
			ep_data->pvt
				= kmalloc(sizeof(struct usbg_se_ep_data),
					  GFP_ATOMIC);
			if(ep_data->pvt){
				memset(ep_data->pvt, 0,
				       sizeof(struct usbg_se_ep_data));
			}
		} else {
			/* terminal	*/
			ep_data->pvt
				= kmalloc(sizeof(struct usbg_te_ep_data),
					  GFP_ATOMIC);
			
			if(ep_data->pvt){
				memset(ep_data->pvt, 0,
				       sizeof(struct usbg_te_ep_data));
			    ep_data_te = (struct usbg_te_ep_data*)ep_data->pvt;
			    ep_data_te->queue_cnt = 0;
			    udif_mutex_init(&ep_data_te->queue_cnt_lock);
			    udif_wait_init(&ep_data_te->wait_dequeue);
			}
		}

		if(!ep_data || !ep_data->pvt){
			err = -ENOBUFS;
			KERR("%s() NULL pointer ep_data=%p, e@_data->pvt=%p\n", __FUNCTION__, ep_data, ep_data->pvt);
			goto SUB_RET;
		}
		
		p_ep->driver_data = ep_data;

		init_waitqueue_head(&ep_data->wait_que);
		ep_data->wait_flag = 0;
		/* request֥Ȥ */
		ep_data->usb_req = usb_ep_alloc_request(p_ep, GFP_ATOMIC);
		if(ep_data->usb_req == NULL){
            KERR("%s() ep_data->usb_req == NULL\n", __FUNCTION__);
			err = -ENOBUFS;
			goto SUB_RET;
		}
	}
	if(list.num_ep==USBG_EPID_NBROF){
		if(usbg_ldisc_start(p_this)){
    		err = -EBUSY;
			KERR("%s() usbg_ldisc_start() failed.\n", __FUNCTION__);
    		goto SUB_RET;
		}
	}
	/* <framework>	*/
	usb_event_add_queue(
		USB_EVENT_PRI_NORMAL,
		p_this->info.event.start,
		(usb_hndl_t)(p_this->info.hndl),
		USBG_KEVENT_ID_CMN_START,
		0,
		(void*)NULL );
	/* </framework>	*/
	err = 0;

SUB_RET:
	if(err){
		cleanup_ep_data(list.ep[USBG_EPID_SE_TX]);
		cleanup_ep_data(list.ep[USBG_EPID_SE_RX]);
		usbg_ldisc_cleanup_ep_data(list.ep[USBG_EPID_TE_TX]);
		usbg_ldisc_cleanup_ep_data(list.ep[USBG_EPID_TE_RX]);
	}
	
	USBG_SEN_EP_SE_LOCK_OFF();

	return;
}

/*  */
/* stop									*/
/*  */
static void core_stop( struct usb_gadget_func_driver* p_drv )
{
	struct func_data* p_this = (struct func_data*)(p_drv->context);
	struct usb_ep* ep;
	struct usbg_sen_ep_data* ep_data;
	int i;

	USBG_SEN_EP_SE_LOCK_ON();

	/* <user>	*/
	
	if(p_this->uc_ep_num==USBG_EPID_NBROF){
		usbg_ldisc_stop(p_this);
	}
	
	for( i=0; i<p_this->uc_ep_num; i++ ){
		ep=p_this->ptbl_ep[i];
		ep_data = ep->driver_data;
		if(ep_data==NULL)
			continue;
		
		set_bit( USBG_SE_FLAG_XFER_INTR, &ep_data->wait_flag );
		wake_up_interruptible(&ep_data->wait_que);

		if(i==USBG_EPID_SE_TX
		   || i==USBG_EPID_SE_RX){
			/* senser	*/
			cleanup_ep_data(ep);
		} else {
			/* terminal	*/
			usbg_ldisc_cleanup_ep_data(ep);
		}
		
		clear_bit(USBG_SE_FLAG_XFER_REQ, &ep_data->wait_flag);
	}


	/* </user>	*/

	/* <framework>	*/
	usbg_cmn_clear_ep_list( p_this );
	/* </framework>	*/

	USBG_SEN_EP_SE_LOCK_OFF();

	/* <framework>	*/
	usb_event_add_queue(
		USB_EVENT_PRI_NORMAL,
		p_this->info.event.stop,
		(usb_hndl_t)(p_this->info.hndl),
		USBG_KEVENT_ID_CMN_STOP,
		0,
		(void*)NULL );
	/* </framework>	*/
}

/*  */
/* vendor								*/
/*  */
static int core_vendor(struct usb_gadget_func_driver* p_drv,
		       const struct usb_ctrlrequest* request, struct usb_ep* ep)
{
	struct func_data* p_this = (struct func_data*)(p_drv->context);
	struct usbg_se_user* p_user = (struct usbg_se_user*)(p_this->vp_user);
	usbg_sen_cmd_id cmd_id = USBG_SEN_CMD_NONE;
	int i;
	int ret=0;
	
	if(p_user->p_cmd==NULL){
		KERR("%s() p_user->p_cmd==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}

	for(i=0;i<USBG_SEN_CMD_NR;i++){
		if(0==memcmp(&p_user->p_cmd[i].setup,
			     request, sizeof(struct usb_ctrlrequest))){
            USBG_SEN_API_DETAIL(" i = %d\n", i);
			ret = usb_gadgetcore_stop_other_driver(
				&p_this->core_reg_info);
			if(ret){
				KERR("%s() error ret=%d\n", __FUNCTION__, ret);
				goto SUB_RET;
			}

			cmd_id = i;
			usb_event_add_queue(
				USB_EVENT_PRI_NORMAL,
				p_this->info.event.vendor_req,
				(usb_hndl_t)(p_this->info.hndl),
				USBG_KEVENT_ID_CMN_VENDOR,
				sizeof(cmd_id),
				(void*)&cmd_id );
			break;
		}
	}

SUB_RET:
	return USB_GADGETCORE_THRU;
}

static int alloc_usbg_se_user(struct usbg_se_user* p_user)
{
	int ret = -ENOSYS;
	if(p_user==NULL){
        KERR("%s() p_user==NULL\n", __FUNCTION__);
		goto SUB_RET;
	}
	p_user->p_cmd = kmalloc(sizeof(struct usbg_sen_cmd)*USBG_SEN_CMD_NR,
				GFP_ATOMIC);
	if(p_user->p_cmd)
		ret = 0;
SUB_RET:
	if(ret)
		cleanup_usbg_se_user(p_user);
	return ret;
}
static void cleanup_usbg_se_user(struct usbg_se_user* p_user)
{
	if(p_user){
		if(p_user->p_cmd){
			kfree(p_user->p_cmd);
			p_user->p_cmd=NULL;
		}
	}
}

static void cleanup_ep_data( struct usb_ep *ep )
{
	struct usbg_sen_ep_data *ep_data;

	if(!ep) return;
	ep_data = ep->driver_data;

	if(!ep_data) return;
    USBG_SEN_INF(" Entry\n");
	/* dequeue */

	if(test_and_clear_bit(USBG_SE_FLAG_USING, &ep_data->wait_flag)){
        USBG_SEN_API_DETAIL(" usb_ep_dequeue( %p, %p) size = %d\n",ep, ep_data->usb_req, ep_data->usb_req->length);
		usb_ep_dequeue(ep, ep_data->usb_req);
	}

	/* usb_request֥Ⱥ                  */
	if(ep_data->usb_req){
		usb_ep_free_request(ep, ep_data->usb_req);
	}

	if(ep_data->pvt){
		kfree(ep_data->pvt);
		ep_data->pvt=NULL;
	}
	
	ep->driver_data=NULL;

}


struct usb_ep* usbg_sen_get_ep( struct func_data* p_this, int num ){
	if(p_this && p_this->ptbl_ep){
		return p_this->ptbl_ep[num];
	} else {
		return NULL;
	}
}

static int setup_buf(void)
{
	int err, fifo_size, total_buf_size;

	err = 0;
#ifdef SENSER_KMALLOC_XFERBUF
    P_SEN_TX_BUF = kmalloc(DEF_SEN_XFER_REQBUF_SZ, GFP_KERNEL);
    total_buf_size = DEF_SEN_XFER_REQBUF_SZ;
#else
    sen_xfer_buf_size /= 16;
    sen_xfer_buf_size *= 16;
    P_SEN_TX_BUF = (unsigned char*)PHYS_TO_CACHE(sen_xfer_buf);
    total_buf_size = sen_xfer_buf_size;
#endif
	if(P_SEN_TX_BUF==NULL){
        KERR( "%s() P_SEN_TX_BUF==NULL\n", __FUNCTION__ );
		err = -ENOMEM;
        goto bail;
	}
	s_TX_BUF_SIZE = total_buf_size >> 2;

#ifdef SENSER_KMALLOC_XFERBUF
    P_SEN_RX_FIFO_BUF = kmalloc(DEF_SEN_XFER_REQBUF_SZ, GFP_KERNEL);
    fifo_size = DEF_SEN_XFER_REQBUF_SZ;
#else
    P_SEN_RX_FIFO_BUF = P_SEN_TX_BUF + (sen_xfer_buf_size / 2);
    fifo_size = sen_xfer_buf_size / 2;
#endif
	if(P_SEN_RX_FIFO_BUF==NULL){
        KERR( "%s() P_SEN_RX_FIFO_BUF==NULL\n", __FUNCTION__ );
#ifdef SENSER_KMALLOC_XFERBUF
        kfree(P_SEN_TX_BUF);
#endif
		err = -ENOMEM;
        goto bail;
	}
    sen_fifo_setup( &s_rx_fifo, P_SEN_RX_FIFO_BUF, fifo_size);

bail:
	return err;
}

#if !defined(SENSER_KMALLOC_XFERBUF)
static int _setup_buf(void *arg)
{
	struct usbg_se_data data;
	struct usbg_se_data *pdata = (struct usbg_se_data*)arg;

	if (copy_from_user(&data, pdata, sizeof(struct usbg_se_data))) {
		PERR("\n");
		return -EBUSY;
	}
	/* ignore data.mode */
	sen_xfer_buf = data.cp;
	sen_xfer_buf_size = data.count;
	return setup_buf();
}
#endif

/**
  *	start_module	-	
  */
static int __init start_module(void)
{
	int err = 0;
	int i;

	err = usbg_cmn_reg(&sonymisc_se);
	err = usbg_cmn_reg(&sonymisc_te);

	USBG_SEN_LOCK_INIT();

	err = usbg_ldisc_init();

	if(err){
		KERR( "%s() module load %d\n", __FUNCTION__, err );
		goto SUB_RET;
	}

	err = setup_buf();
	if (err)
		goto SUB_RET;

	for(i=0;i<USBG_EPID_NBROF;i++){
		ptbl_ep_data[i]
			= kmalloc(sizeof(struct usbg_sen_ep_data),GFP_KERNEL);
	}
	
SUB_RET:
	return err;
}
module_init(start_module);

/**
  *	stop_module	-	
  */
static void __exit stop_module(void)
{
	int i;
#ifdef SENSER_KMALLOC_XFERBUF
	if(P_SEN_TX_BUF){
		kfree(P_SEN_TX_BUF);
		P_SEN_TX_BUF=NULL;
	}
    if( P_SEN_RX_FIFO_BUF){
        kfree(P_SEN_RX_FIFO_BUF);
        P_SEN_RX_FIFO_BUF = NULL;
    }
#endif
	for(i=0;i<USBG_EPID_NBROF;i++){
		if(ptbl_ep_data[i])
			kfree(ptbl_ep_data[i]);
		ptbl_ep_data[i]=NULL;
	}
	
	usbg_ldisc_exit();
	usbg_cmn_dereg(&sonymisc_se);
	usbg_cmn_dereg(&sonymisc_te);
}
module_exit(stop_module);

