/**
 *	@file	usbg_still_evt.c
 *	@brief	USB SICD(Still Image Capture Device class) - event
 *	
 *		Copyright 2005,2006,2011 Sony Corporation
 * Copyright 2018, 2019 Sony Imaging Products and Solutions Incorporated.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <asm/errno.h>
#include <asm/uaccess.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>

#include <linux/device.h>

#include <linux/usb/gadget.h>

#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#ifndef CONFIG_OSAL_UDIF
#include <asm/arch/sonymisc.h>
#endif

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_still.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>

#define FNAME "usbg_still_evt.c"
#include "usbg_still_conf.h"
#include "usbg_still_pvt.h"

//#define PTP_EVENT_BOOTLOG

#ifdef PTP_EVENT_BOOTLOG
#include <linux/snsc_boot_time.h>
#endif

/**
 *	@brief		SICD event driver open
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 */
UDIF_ERR sicd_event_open(UDIF_FILE *filp)
{
	struct file fd;
	struct file *file = &fd;
	SICD_API();
	sicd_link_this_to_file(file);
    udif_file_data(filp) = file->private_data;
	return UDIF_ERR_OK;
}

/**
 *	@brief		SICD event driver release
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 */
UDIF_ERR sicd_event_release(UDIF_FILE *filp)
{
	SICD_API();
	return UDIF_ERR_OK;
}


/**
 *	@brief		SICD event write(send) complete callback
 *	@param		ep			bulk-in ep
 *	@param		req			request struct
 */
static void sicd_evt_comp(struct usb_ep *ep, struct usb_request *req)
{
    
    struct sicd_evt_req *req_info = req->context;
    
	SICD_API();
	sicd_free_evt_req_info(ep, req_info);
	sicd_recheck_req_comp_waiting();
#ifdef PTP_EVENT_BOOTLOG
	{
		unsigned char buf[20];
		snprintf(buf, sizeof(buf), "evt_cmp:%d", req_info->index);
		boot_time_add(buf);
	}
#endif
}


/**
 *	@brief		SICD event write(send)
 *	@param		file		file struct
 *	@param		data		data buffer
 *	@param		size		data size
 *	@param		off			offset
 *	@retval		0			success
 *	@retval		-EINVAL		invalid parameter
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EIO		I/O error
 */
UDIF_SSIZE sicd_event_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param)
{
	struct func_data_ex *p_this_ex = sicd_get_this_ex();
	const char *data = param->uaddr;
	size_t size = param->count;
	int err;
    
    struct sicd_evt_req *req_info;
	struct usb_request *req;
	struct usb_ep *ep;
	
	SICD_API();
	
	// param check
	if(!data ||
	    size > SICD_EVT_BUFSIZE) {
		SICD_ERR("data==NULL\n");
		return UDIF_ERR_PAR;
	}
	
	SICD_EP_LOCK_ON();
	
	if (!p_this_ex->bulk_evt_mode) {
		ep = sicd_get_ep(SICD_EP_IDX_INT_IN);
	} else {
		ep = sicd_get_ep(SICD_EP_IDX_BULK_IN);
	}
	if(!ep) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("ep==NULL\n");
		return UDIF_ERR_IO;
	}
	
	// alloc request
	req_info = sicd_alloc_evt_req_info(ep);
	if(!req_info) {
#ifdef PTP_EVENT_BOOTLOG
		boot_time_add("evt_dis");
#endif
		SICD_EP_LOCK_OFF();
		SICD_ERR("can not get request\n");
		return UDIF_ERR_NOMEM;
	}
    req = req_info->req;
	
	err = copy_from_user(req_info->evt_buffer, data, size);
	if(err) {
		SICD_EP_LOCK_OFF();
		SICD_ERR("copy_from_user\n");
		return UDIF_ERR_BUSY;
	}
	
	// fill request  (send NULL packet)
	req->buf			= req_info->evt_buffer;
	req->length			= size;
	req->dma			= (dma_addr_t)NULL;
	req->zero			= req->length ? 0 : 1;	// append NULL packet if needed
	req->complete		= sicd_evt_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= req_info;
	
#ifdef PTP_EVENT_BOOTLOG
	{
		unsigned char buf[20];
		snprintf(buf, sizeof(buf), "evt_que:%d", req_info->index);
		boot_time_add(buf);
	}
#endif

	// queue request
	err = usb_ep_queue(ep, req, GFP_ATOMIC);
	
	SICD_EP_LOCK_OFF();
	
	if(err) {
		SICD_ERR("usb_ep_queue err_code=%d\n", err);
		sicd_free_evt_req_info(ep, req_info);
		return err;
	}
	
	return req->length;	
}
