/**
 *	@file	usbg_iap_core.c
 *	@brief	USB iAP(iAP Device Interface) - core
 *	
 *	Copyright 2024 Sony Group Corporation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <asm/uaccess.h>
#include <asm/page.h>
#include <asm/errno.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>

#include <linux/device.h>

#include <linux/usb/gadget.h>

#include <linux/mm.h>
#include <linux/usb/gcore/usb_event.h>
#include <linux/usb/gcore/usb_gadgetcore.h>

#include <linux/usb/specific_gadget/usbg_cmn.h>
#include <linux/usb/specific_gadget/usbg_type.h>
#include <linux/usb/specific_gadget/usbg_iap.h>
#include <linux/usb/specific_gadget/usb_kdbg.h>
#include <linux/usb/specific_gadget/usb_ioctl.h>

#include <udif/malloc.h>

#define FNAME "usbg_iap_core.c"	///< this file name
#include "usbg_iap_conf.h"
#include "usbg_iap_pvt.h"

static void core_cb_start( struct usb_gadget_func_driver* p_drv,
						unsigned char alt,
						struct usb_gadget_ep_list list )
{
	struct func_data    *p_this    = iap_get_this();
	struct func_data_ex *p_this_ex = iap_get_this_ex();
	struct usb_ep *ep;
	struct iap_ep_info *epinf;
	
	IAP_API();

	IAP_EP_LOCK_ON();

	if (p_this_ex->activated) {
		IAP_EP_LOCK_OFF();
		IAP_INF("already started, do nothing\n");
		return;
	}
	if (list.num_ep != IAP_EP_IDX_MAX) {
		IAP_EP_LOCK_OFF();
		IAP_INF("num_ep out of range %d\n", list.num_ep);
		return;
	}
	p_this_ex->activated = true;

	if( usbg_cmn_make_ep_list( p_this, alt, list ) ){
		IAP_ERR("usbg_cmn_make_ep_list\n");
	}
	
	// ep table initialize
	ep                       = p_this->ptbl_ep[IAP_EP_IDX_BULK_IN];
	epinf                    = &(p_this_ex->ep_info[IAP_EP_IDX_BULK_IN]);
	epinf->ep_no             = 0x84;
	epinf->stalled           = false;
	epinf->data.requesting   = false;
	epinf->data.req          = usb_ep_alloc_request(ep, GFP_ATOMIC);
	ep->name                 = IAP_EP_NAME_BULK_IN;
	ep->driver_data          = epinf;
	
	ep                       = p_this->ptbl_ep[IAP_EP_IDX_BULK_OUT];
	epinf                    = &(p_this_ex->ep_info[IAP_EP_IDX_BULK_OUT]);
	epinf->ep_no             = 0x05;
	epinf->stalled           = false;
	epinf->data.requesting   = false;
	epinf->data.req          = usb_ep_alloc_request(ep, GFP_ATOMIC);
	ep->name                 = IAP_EP_NAME_BULK_OUT;
	ep->driver_data          = epinf;
	
	IAP_EP_LOCK_OFF();
	
	usb_event_add_queue(
			USB_EVENT_PRI_NORMAL,
			p_this->info.event.start,
			p_this->info.hndl,
			USBG_KEVENT_ID_CMN_START,
			0,
			NULL
	);
	IAP_INF("USBG_KEVENT_ID_CMN_START\n");
}

/**
 *	@brief		GadgetCore callback - stop
 *	@param		p_drv		function driver struct
 */
static void core_cb_stop( struct usb_gadget_func_driver* p_drv )
{
	struct func_data    *p_this    = iap_get_this();
	struct func_data_ex *p_this_ex = iap_get_this_ex();

	IAP_API();

	IAP_EP_LOCK_ON();

	if (!p_this_ex->activated) {
		IAP_EP_LOCK_OFF();
		IAP_INF("already stopped, do nothing\n");
		return;
	}
	p_this_ex->activated = false;

	iap_ep_dequeue_all();

	if (iap_check_req_comp_waiting()) {
		IAP_EP_LOCK_OFF();
		if (wait_for_completion_interruptible(&(p_this_ex->req_wait_completion))) {
			IAP_INF("interrupted completion\n");
		}
		IAP_EP_LOCK_ON();
	}

	// ep info initalize
	usb_ep_free_request(p_this->ptbl_ep[IAP_EP_IDX_BULK_OUT],
						p_this_ex->ep_info[IAP_EP_IDX_BULK_OUT].data.req);

	usb_ep_free_request(p_this->ptbl_ep[IAP_EP_IDX_BULK_IN],
						p_this_ex->ep_info[IAP_EP_IDX_BULK_IN].data.req);

	memset(p_this_ex->ep_info, 0, sizeof(struct iap_ep_info) * IAP_EP_IDX_MAX);

	usbg_cmn_clear_ep_list(p_this);

	kfree(p_this_ex->in_buffer);
	p_this_ex->in_buffer = NULL;
	kfree(p_this_ex->out_buffer);
	p_this_ex->out_buffer = NULL;

	IAP_EP_LOCK_OFF();
	
	usb_event_add_queue(
			USB_EVENT_PRI_NORMAL,
			p_this->info.event.stop,
			p_this->info.hndl,
			USBG_KEVENT_ID_CMN_STOP,
			0,
			NULL
	);
	IAP_INF("USBG_KEVENT_ID_CMN_STOP\n");
}

/**
 *	@brief		IAP core ioctl - probe
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 */
static int ioctl_probe(struct file *file, unsigned long arg)
{
	int err;
	struct usb_gadget_func_driver* reginfo;
	struct probe_info	*p_info    = (struct probe_info *)arg;
	struct func_data	*p_this    = iap_get_this();
	int ifnum;

	IAP_FNC();
	
	if(usbg_cmn_copy_probe_info(p_this, p_info)){
		IAP_ERR("copy probe info\n");
		return -ENOBUFS;
	}
	
	ifnum = p_this->info.uc_func_info_interface_num;
	reginfo     = &(p_this->core_reg_info);
	reginfo->function		= IAP_FUNC_NAME;
	reginfo->context		= p_this;
	reginfo->start			= core_cb_start;
	reginfo->stop			= core_cb_stop;
	
	reginfo->config = 
		p_this->info.tbl_interface[0].uc_configuration;
	reginfo->interface = 
		p_this->info.tbl_interface[0].uc_interface;
	
	IAP_INF("register %s, cfg=%d, if=%d\n",
				reginfo->function,
				reginfo->config,
				reginfo->interface);
				
	err = usb_gadgetcore_register_driver(reginfo);
	if(err) {
		IAP_INF("usb_gadgetcore_register_driver error=%d\n",err);
	}
	
	return err;
}

/**
 *	@brief		IAP core ioctl - remove
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 */
static int ioctl_remove(struct file *file, unsigned long arg)
{
	int ret;
	struct func_data* p_this = iap_get_this();

	IAP_FNC();

	ret = usb_gadgetcore_unregister_driver(&(p_this->core_reg_info));
	usbg_cmn_clear_probe_info(p_this);
	
	return ret;
}

/**
 *	@brief		IAP core ioctl - Gadget functin driver common
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOBUFS	no buffer
 *	@retval		-EFAULT		segmentation fault
 *	@retval		-ENOTTY		invalid ioctl no.
 */
static int ioctl_cmn(struct file *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;

//	IAP_FNC();

	// valid ioctl no?
	if( _IOC_NR(cmd) > USB_IOC_CMN_NBROF ){
		IAP_ERR("invalid ioctl seq. number\n");
		return -ENOTTY;
	}
	
	// read/write access?
	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd) );
	} else if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd) );
	}
	
	// access ok?
	if(err){
		IAP_ERR("invalid ioctl dir\n");
		return -EFAULT;
	}
	
	// ioctl for what?
	switch (cmd) {
		case USB_IOC_CMN_PROBE:
			err = ioctl_probe(file, arg);
			break;
		case USB_IOC_CMN_REMOVE:
			err = ioctl_remove(file, arg);
			break;
		default:
			IAP_ERR("invalid ioctl seq. number\n");
			err = -ENOTTY;
			break;
	}
	
	return err;
}

/**
 *	@brief		IAP core driver open
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 */
UDIF_ERR iap_core_open(UDIF_FILE *filp)
{
	struct file fd;
	struct file *file = &fd;

	IAP_API();

	iap_link_this_to_file(file);
    udif_file_data(filp) = file->private_data;

	return UDIF_ERR_OK;
}

/**
 *	@brief		IAP core driver release
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@retval		0			success
 */
UDIF_ERR iap_core_release(UDIF_FILE *filp)
{
	IAP_API();

	return UDIF_ERR_OK;
}

/**
 *	@brief		IAP core ioctl - get read data
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 */
static int ioctl_iap_get_data(struct file *file, unsigned long arg)
{
	int err;
	struct usbg_iap_data st;
	struct func_data_ex *p_this_ex = iap_get_this_ex();

	err = copy_from_user(&st, (struct usbg_iap_data *)arg, sizeof(struct usbg_iap_data));
	if(err > 0) {
		IAP_ERR("copy_from_user failed n=%d\n", err);
		return -ENOBUFS;
	}

	if (st.read_data && (st.buf_sz >= p_this_ex->out_used_size)) {
		err = copy_to_user(st.read_data, p_this_ex->out_buffer, p_this_ex->out_used_size);
		if(err > 0) {
			IAP_ERR("copy_to_user failed n=%d\n", err);
			return -ENOBUFS;
		}
	} else {
		IAP_ERR("no space error\n");
		return -ENOBUFS;
	}

	return p_this_ex->out_used_size - err;
}

/**
 *	@brief		IAP core ioctl
 *	@param		file		file struct
 *	@param		arg			ioctl arguments
 */
static int ioctl_iap(struct file *file, unsigned int cmd, unsigned long arg)
{
	int err = 0;

	// read/write access?
	if( _IOC_DIR(cmd) & _IOC_READ ){
		err = !access_ok(VERIFY_WRITE, (void *)arg, _IOC_SIZE(cmd) );
		if(err){
			IAP_ERR("invalid ioctl dir\n");
			return -EFAULT;
		}
	}
	if( _IOC_DIR(cmd) & _IOC_WRITE ){
		err = !access_ok(VERIFY_READ, (void *)arg, _IOC_SIZE(cmd) );
		if(err){
			IAP_ERR("invalid ioctl dir\n");
			return -EFAULT;
		}
	}

	// ioctl for what?
	switch (cmd) {
		case USBG_IOC_IAP_GET_DATA:
			err = ioctl_iap_get_data(file, arg);
			break;
		default:
			IAP_ERR("invalid ioctl seq. number\n");
			err = -ENOTTY;
			break;
	}

	return err;
}

/**
 *	@brief		IAP core ioctl
 *	@param		inode		inode struct
 *	@param		file		file struct
 *	@param		cmd			ioctl no.
 *	@param		arg			ioctl arguments
 *	@retval		0			success
 *	@retval		-ENOTTY		wrong ioctl no.
 */
UDIF_ERR iap_core_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param)
{
	struct file fd;
	struct file *file = &fd;
	unsigned int cmd = param->cmd;
	unsigned long arg = param->arg;
	int err;

//	IAP_API();
	
	file->private_data = udif_file_data(filp);
	if( _IOC_TYPE(cmd) == USB_IOC_CMN ){
		// common ioctl
		err = ioctl_cmn(file, cmd, arg );
	}
	else if( _IOC_TYPE(cmd) == USBG_IOC_IAP ){
		err = ioctl_iap(file, cmd, arg );
	}
	else {
		// invalid ioctl
		IAP_ERR("invalid ioctl magic number\n");
		err = UDIF_ERR_NOTTY;
	}
	udif_file_data(filp) = file->private_data;
	
	return (UDIF_ERR)err;
}

/**
 *	@brief		IAP data write(send) complete callback
 *	@param		ep			bulk-in ep
 *	@param		file		request struct
 *	@retval		none
 */
static void iap_core_write_comp(struct usb_ep *ep, struct usb_request *req)
{
	struct func_data *p_this = iap_get_this();
	struct USBG_IAP_KEVENT_FUNC *event_func = p_this->info.vp_user;
	
	IAP_API();
	
	IAP_INF("status=%d, length=%d, actual=%d\n", req->status, req->length, req->actual);
	
	iap_request_stop( IAP_EP_IDX_BULK_IN );
	
	iap_recheck_req_comp_waiting();
	
	usb_event_add_queue(
			USB_EVENT_PRI_NORMAL,
			event_func->write_comp,
			p_this->info.hndl,
			USBG_IAP_KEVENT_WRITE_COMP,
			0,
			NULL
	);
	IAP_INF("USBG_IAP_KEVENT_WRITE_COMP\n");

}

/**
 *	@brief		IAP data write(send)
 *	@param		file		file struct
 *	@param		data		data buffer
 *	@param		size		data size
 *	@param		off			offset
 *	@retval		0			success
 *	@retval		-EINVAL		invalid parameter
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EIO		I/O error
 */
UDIF_SSIZE iap_core_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param)
{
	const char *data = param->uaddr;
	size_t size = param->count;
	int err;
	struct usb_request *req;
	struct usb_ep *ep;

	struct func_data_ex *p_this_ex = iap_get_this_ex();
	
	IAP_API();
	
	IAP_EP_LOCK_ON();

	if( !p_this_ex->activated ) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("access denied\n");
		return UDIF_ERR_ACCES;
	}

	ep = iap_get_ep(IAP_EP_IDX_BULK_IN);
	if(!ep) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("ep==NULL\n");
		return UDIF_ERR_IO;
	}

	if( iap_is_requesting(IAP_EP_IDX_BULK_IN) ) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("ep busy\n");
		return UDIF_ERR_BUSY;
	}
	
	// alloc request
	req = iap_get_request(ep);
	if(!req) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("can not alloc request\n");
		return UDIF_ERR_NOMEM;
	}
	
	if (size == 0 || size > IAP_DAT_BUFSIZE) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("size error\n");
		return UDIF_ERR_NOMEM;
	}

	if (!p_this_ex->in_buffer) {
		p_this_ex->in_buffer = kzalloc(IAP_DAT_BUFSIZE, GFP_KERNEL);
	}

	err = copy_from_user(p_this_ex->in_buffer, data, size);
	if (err != 0) IAP_ERR("copy_from_user err=%d\n", err);

	// fill request
	req->buf			= p_this_ex->in_buffer;
	req->length			= size;
	req->complete		= iap_core_write_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= (void*)data;
	req->dma			= (dma_addr_t)NULL;
	
	if(req->length % ep->maxpacket == 0) {
		req->zero = 1;	// append NULL packet
	}
	else {
		req->zero = 0;
	}

	if(!req->buf) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("buffer area err\n");
		return UDIF_ERR_FAULT;
	}
	
	iap_request_start( IAP_EP_IDX_BULK_IN );

	// queue request
	err = usb_ep_queue(ep, req, GFP_ATOMIC);

	IAP_EP_LOCK_OFF();

	if(err) {
		// usb_ep_queue error
		IAP_ERR("usb_ep_queue err_code=%d\n", err);
		iap_request_stop( IAP_EP_IDX_BULK_IN );
		return err;
	}
	
	return req->length;
}

/**
 *	@brief		IAP data read(receive) complete callback
 *	@param		ep			bulk-out ep
 *	@param		file		request struct
 *	@retval		none
 */
static void iap_core_read_comp(struct usb_ep *ep, struct usb_request *req)
{
	struct func_data *p_this = iap_get_this();
	struct func_data_ex *p_this_ex = iap_get_this_ex();
	struct USBG_IAP_KEVENT_FUNC *event_func = p_this->info.vp_user;
	
	IAP_API();
	
	IAP_INF("status=%d, length=%d, actual=%d\n", req->status, req->length, req->actual);
    
	iap_request_stop( IAP_EP_IDX_BULK_OUT );

	iap_recheck_req_comp_waiting();

	if (req->actual > 0) {
		p_this_ex->out_used_size = req->actual;
	} else {
		p_this_ex->out_used_size = 0;
	}

	usb_event_add_queue(
			USB_EVENT_PRI_NORMAL,
			event_func->read_comp,
			p_this->info.hndl,
			USBG_IAP_KEVENT_READ_COMP,
			sizeof(p_this_ex->out_used_size),
			&p_this_ex->out_used_size
	);
	IAP_INF("USBG_IAP_KEVENT_READ_COMP\n");
}

/**
 *	@brief		IAP data read(receive)
 *	@param		file		file struct
 *	@param		data		data buffer
 *	@param		size		data size
 *	@param		off			offset
 *	@retval		0			success
 *	@retval		-EINVAL		invalid parameter
 *	@retval		-ENOMEM		short of memory
 *	@retval		-EIO		I/O error
 */
UDIF_SSIZE iap_core_read(UDIF_FILE *filp, UDIF_CDEV_READ *param)
{
	char *data = param->uaddr;
	size_t size = param->count;
	int err;
	struct usb_request *req;
	struct usb_ep *ep;
	struct func_data_ex *p_this_ex = iap_get_this_ex();
	
	IAP_API();

	IAP_EP_LOCK_ON();

	if( !p_this_ex->activated ) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("access denied\n");
		return UDIF_ERR_ACCES;
	}

	ep = iap_get_ep(IAP_EP_IDX_BULK_OUT);
	if(!ep) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("ep==NULL\n");
		return UDIF_ERR_IO;
	}

	if( iap_is_requesting(IAP_EP_IDX_BULK_OUT) ) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("ep busy\n");
		return UDIF_ERR_BUSY;
	}
	
	// alloc request
	req = iap_get_request(ep);
	if(!req) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("can not alloc request\n");
		return UDIF_ERR_NOMEM;
	}

	if (size == 0 || size > IAP_DAT_BUFSIZE) {
		IAP_EP_LOCK_OFF();
		IAP_ERR("size error\n");
		return UDIF_ERR_NOMEM;
	}

	if (!p_this_ex->out_buffer) {
		p_this_ex->out_buffer = kzalloc(IAP_DAT_BUFSIZE, GFP_KERNEL);
	}

	// fill request
	req->buf			= p_this_ex->out_buffer;
	req->length			= size;
	IAP_INF("%s: size=%ld, aligned=%d\n", __FUNCTION__, size, req->length);
	req->zero			= 0;
	req->complete		= iap_core_read_comp;
	req->no_interrupt	= 0;
	req->short_not_ok	= 0;
	req->context		= (void*)data;
	req->dma			= (dma_addr_t)NULL;

	iap_request_start( IAP_EP_IDX_BULK_OUT );

	// queue request
	err = usb_ep_queue(ep, req, GFP_ATOMIC);

	IAP_EP_LOCK_OFF();

	if(err) {
		// usb_ep_queue error
		IAP_ERR("usb_ep_queue err_code=%d\n", err);
		iap_request_stop( IAP_EP_IDX_BULK_OUT );
		return err;
	}
	
	return 0;
}
