/*
 * Copyright 2018 Sony Imaging Products & Solutions Inc.
 * Copyright 2018 Sony Corporation.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/mfd/syscon.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/of_address.h>
#include <linux/clk.h>
#include <linux/clk-provider.h>
#include <linux/regmap.h>
#include <linux/spinlock.h>
#include <linux/moduleparam.h>
#include <dt-bindings/clock/cxd-900xx-clk.h>

#include "cxd-clk-define.h"
#include "cxd-clk.h"

#ifdef CONFIG_DEBUG_FS
#include "cxd-clk-debugfs.h"
#endif

static int debug;
module_param(debug, int, S_IRUSR|S_IWUSR);

static char *ver = "2019_0109_00";
module_param(ver, charp, 0400);

static struct cxd_clk_ctrl_data cxd_clk;

static struct cxd_clk_fix_node cxd900xx_clk_fix[] = {
        {CXD_CLK_FIX_SYS_IN, "fix_sys_in", RATE_24M},
        {CXD_CLK_FIX_PLL0_OUT, "fix_pll0", RATE_108M},
        {CXD_CLK_FIX_PLL0_CK396, "fix_pll3_396", RATE_396M},
        {CXD_CLK_FIX_PLL0_CK198, "fix_pll3_198", RATE_198M},
        {CXD_CLK_FIX_PLL0_CK99, "fix_pll3_99", RATE_99M},
        {CXD_CLK_FIX_PLL0_CK49P5, "fix_pll3_49_5", RATE_49_5M},
        {CXD_CLK_FIX_PLL0_CK24P75, "fix_pll3_24_75", RATE_24_75M},
        {CXD_CLK_FIX_PLL4_OUT, "fix_pll4", RATE_1536M},
        {CXD_CLK_FIX_PLL4_CK_192, "fix_pll4_192", RATE_192M},
        {CXD_CLK_FIX_PLL4_CK_96, "fix_pll4_96", RATE_96M},
        {CXD_CLK_FIX_PLL4_CK_48, "fix_pll4_48", RATE_48M},
        {CXD_CLK_FIX_PLL4_CK_12, "fix_pll4_12", RATE_12M},
        {CXD_CLK_FIX_PLL4_CK_2, "fix_pll4_4", RATE_4M},
        {CXD_CLK_FIX_PLL4_CK_2, "fix_pll4_2", RATE_2M},
        {CXD_CLK_FIX_PLL4_CK_1, "fix_pll4_1", RATE_1M},
        {CXD_CLK_FIX_PLL4_CK_400K, "fix_pll4_400k", RATE_400K},
        {CXD_CLK_FIX_PLL4_CK_33K, "fix_pll4_33k", RATE_33K},
};

static struct cxd_clk_div_node cxd900xx_clk_div[] = {
        {CXD_CLK_DIV_SDIO, "div_sdio", "fix_pll4", 0x10C, 0, 8},
        {CXD_CLK_DIV_EMMC, "div_emmc_boot", "fix_pll4", 0x110, 0, 8},
        {CXD_CLK_DIV_SDIF0, "div_sdif0", "fix_pll4", 0x114, 0, 8},
        {CXD_CLK_DIV_SDIF1, "div_sdif1", "fix_pll4", 0x118, 0, 8},
        {CXD_CLK_DIV_MS, "div_ms", "fix_pll4", 0x11C, 0, 8},
};

static const char * const uart_sels[] = {"fix_pll4_12", "fix_pll4_48",};
static const char * const spi_sels[] = {"fix_pll4_48", "fix_pll4_96", "fix_pll4_192", "fix_pll4_48",};
static const char * const sdio_sels[] = {"fix_pll4_400k", "div_sdio",};
static const char * const emmc_sels[] = {"fix_pll4_400k", "div_emmc_boot",};
static const char * const sdif0_sels[] = {"fix_pll4_400k", "div_sdif0",};
static const char * const sdif1_sels[] = {"fix_pll4_400k", "div_sdif1",};
static const char * const ms_sels[] = {"fix_pll4_400k", "div_ms",};

static struct cxd_clk_mux_node cxd900xx_clk_mux[] = {
        {CXD_CLK_SEL_UART0, "sel_uart0", uart_sels, ARRAY_SIZE(uart_sels), 0x210, 24, 1},
        {CXD_CLK_SEL_UART1, "sel_uart1", uart_sels, ARRAY_SIZE(uart_sels), 0x210, 25, 1},
        {CXD_CLK_SEL_UART2, "sel_uart2", uart_sels, ARRAY_SIZE(uart_sels), 0x210, 26, 1},
        {CXD_CLK_SEL_UART3, "sel_uart3", uart_sels, ARRAY_SIZE(uart_sels), 0x210, 27, 1},
        {CXD_CLK_SEL_UART4, "sel_uart4", uart_sels, ARRAY_SIZE(uart_sels), 0x210, 28, 1},
        {CXD_CLK_SEL_SDIO, "sel_sdio", sdio_sels, ARRAY_SIZE(sdio_sels), 0x220, 0, 1},
        {CXD_CLK_SEL_EMMC, "sel_emmc", emmc_sels, ARRAY_SIZE(emmc_sels), 0x220, 4, 1},
        {CXD_CLK_SEL_SDIF0, "sel_sdif0", sdif0_sels, ARRAY_SIZE(sdif0_sels), 0x220, 8, 1},
        {CXD_CLK_SEL_SDIF1, "sel_sdif1",  sdif1_sels, ARRAY_SIZE(sdif1_sels), 0x220, 12, 1},
        {CXD_CLK_SEL_MS, "sel_ms", ms_sels, ARRAY_SIZE(ms_sels), 0x220, 16, 1},
        {CXD_CLK_SEL_SPI0, "sel_spi0", spi_sels, ARRAY_SIZE(spi_sels), 0x230, 0, 2},
        {CXD_CLK_SEL_SPI1, "sel_spi1", spi_sels, ARRAY_SIZE(spi_sels), 0x230, 2, 2},
        {CXD_CLK_SEL_SPI2, "sel_spi2", spi_sels, ARRAY_SIZE(spi_sels), 0x230, 4, 2},
        {CXD_CLK_SEL_SPI3, "sel_spi3", spi_sels, ARRAY_SIZE(spi_sels), 0x230, 6, 2},
        {CXD_CLK_SEL_SPI4, "sel_spi4", spi_sels, ARRAY_SIZE(spi_sels), 0x230, 8, 2},
};

static struct cxd_clk_gate_node cxd900xx_clk_gate[] = {
        {CXD_CLK_GATE_USB_REF_CLK, "gate_usb_ref", "fix_sys_in", 0x00, 12},
        {CXD_CLK_GATE_USB_PCLK, "gate_usb_p", "fix_pll3_99", 0x00, 13},
        {CXD_CLK_GATE_USB_ACLK, "gate_usb_a", "fix_pll3_396", 0x00, 14},
        {CXD_CLK_GATE_USB_SUS_CLK, "gate_usb_sus", "fix_pll4_33k", 0x00, 15},
        {CXD_CLK_GATE_MS, "gate_ms", "sel_ms", 0x00, 16},
        {CXD_CLK_GATE_DMA0, "gate_DMA0", "fix_pll3_396", 0x20, 0},
        {CXD_CLK_GATE_DMA1, "gate_DMA1", "fix_pll3_396", 0x20, 1},
        {CXD_CLK_GATE_DMAPERI0, "gate_DMAPERI0", "fix_pll3_396", 0x20, 2},
        {CXD_CLK_GATE_DMAPERI1, "gate_DMAPERI1", "fix_pll3_396", 0x20, 3},
        {CXD_CLK_GATE_CRYPT_BOOT, "gate_crypt_boot", "fix_pll3_396", 0x20, 8},
        {CXD_CLK_GATE_CRYPT_SPACC, "gate_crypt_spacc", "fix_pll3_396", 0x20, 9},
        {CXD_CLK_GATE_GIC_MAIN, "gate_gic_main", "fix_pll3_396", 0x30, 2},
        {CXD_CLK_GATE_GIC_SUB, "gate_gic_sub", "fix_pll3_396", 0x30, 3},
        {CXD_CLK_GATE_X2V_PCEI0, "gate_x2v_pcie0", NULL, 0x30, 18},
        {CXD_CLK_GATE_X2V_PCEI1, "gate_x2v_pcie1", NULL, 0x30, 19},
        {CXD_CLK_GATE_X2V_MEDIA, "gate_x2v_media", NULL, 0x30, 21},
        {CXD_CLK_GATE_TIMER0, "gate_timer0", "fix_pll4_2", 0x40, 6},
        {CXD_CLK_GATE_TIMER1, "gate_timer1", "fix_pll4_2", 0x40, 7},
        {CXD_CLK_GATE_TIMER2, "gate_timer2", "fix_pll4_2", 0x40, 8},
        {CXD_CLK_GATE_TIMER3, "gate_timer3", "fix_pll4_2", 0x40, 9},
        {CXD_CLK_GATE_TIMER4, "gate_timer4", "fix_pll4_2", 0x40, 10},
        {CXD_CLK_GATE_TIMER5, "gate_timer5", "fix_pll4_2", 0x40, 11},
        {CXD_CLK_GATE_TIMER6, "gate_timer6", "fix_pll4_2", 0x40, 12},
        {CXD_CLK_GATE_TIMER7, "gate_timer7", "fix_pll4_2", 0x40, 13},
        {CXD_CLK_GATE_TIMER8, "gate_timer8", "fix_pll4_2", 0x40, 14},
        {CXD_CLK_GATE_TIMER9, "gate_timer9", "fix_pll4_2", 0x40, 15},
        {CXD_CLK_GATE_TIMER10, "gate_timer10", "fix_pll4_2", 0x40, 16},
        {CXD_CLK_GATE_TIMER11, "gate_timer11", "fix_pll4_2", 0x40, 17},
        {CXD_CLK_GATE_TIMER12, "gate_timer12", "fix_pll4_2", 0x40, 18},
        {CXD_CLK_GATE_TIMER13, "gate_timer13", "fix_pll4_2", 0x40, 19},
        {CXD_CLK_GATE_TIMER14, "gate_timer14", "fix_pll4_2", 0x40, 20},
        {CXD_CLK_GATE_TIMER15, "gate_timer15", "fix_pll4_2", 0x40, 21},
        {CXD_CLK_GATE_WDT, "gate_wdt", "fix_pll4_4", 0x40, 24},
        {CXD_CLK_GATE_UART0, "gate_uart0", "sel_uart0", 0x50, 6},
        {CXD_CLK_GATE_UART1, "gate_uart1", "sel_uart1", 0x50, 7},
        {CXD_CLK_GATE_UART2, "gate_uart2", "sel_uart2", 0x50, 8},
        {CXD_CLK_GATE_UART3, "gate_uart3", "sel_uart3", 0x50, 9},
        {CXD_CLK_GATE_UART4, "gate_uart4", "sel_uart4", 0x50, 10},
        {CXD_CLK_GATE_SPI0, "gate_spi0", "sel_spi0", 0x50, 12},
        {CXD_CLK_GATE_SPI1, "gate_spi1", "sel_spi1", 0x50, 13},
        {CXD_CLK_GATE_SPI2, "gate_spi2", "sel_spi2", 0x50, 14},
        {CXD_CLK_GATE_SPI3, "gate_spi3", "sel_spi3", 0x50, 15},
        {CXD_CLK_GATE_SPI4, "gate_spi4", "sel_spi4", 0x50, 16},
        {CXD_CLK_GATE_I2C0, "gate_i2c0", "fix_pll4_48", 0x50, 18},
        {CXD_CLK_GATE_I2C1, "gate_i2c1", "fix_pll4_48", 0x50, 19},
        {CXD_CLK_GATE_PCIE_I_PCLK, "gate_pcie_i_p", "fix_pll3_99", 0x80, 1},
        {CXD_CLK_GATE_PCIE_I_DBI_CLK, "gate_pcie_i_dbi", "fix_pll3_396", 0x80, 2},
        {CXD_CLK_GATE_PCIE_I_SLV_CLK, "gate_pcie_i_slv", "fix_pll3_396", 0x80, 3},
        {CXD_CLK_GATE_PCIE_I_MST_CLK, "gate_pcie_i_mst", "fix_pll3_396", 0x80, 4},
        {CXD_CLK_GATE_PCIE_I_AUX_CLK, "gate_pcie_i_aux", "fix_pll4_1", 0x80, 5},
        {CXD_CLK_GATE_PCIE_M0_PCLK, "gate_pcie_m0_p", "fix_pll3_99", 0x80, 7},
        {CXD_CLK_GATE_PCIE_M0_DBI_CLK, "gate_pcie_m0_dbi", "fix_pll3_396", 0x80, 8},
        {CXD_CLK_GATE_PCIE_M0_SLV_CLK, "gate_pcie_m0_slv", "fix_pll3_396", 0x80, 9},
        {CXD_CLK_GATE_PCIE_M0_MST_CLK, "gate_pcie_m0_mst", "fix_pll3_396", 0x80, 10},
        {CXD_CLK_GATE_PCIE_M0_AUX_CLK, "gate_pcie_m0_aux", "fix_pll4_1", 0x80, 11},
        {CXD_CLK_GATE_PCIE_M1_PCLK, "gate_pcie_m1_p", "fix_pll3_99", 0x80, 13},
        {CXD_CLK_GATE_PCIE_M1_DBI_CLK, "gate_pcie_m1_dbi", "fix_pll3_396", 0x80, 14},
        {CXD_CLK_GATE_PCIE_M1_SLV_CLK, "gate_pcie_m1_slv", "fix_pll3_396", 0x80, 15},
        {CXD_CLK_GATE_PCIE_M1_MST_CLK, "gate_pcie_m1_mst", "fix_pll3_396", 0x80, 16},
        {CXD_CLK_GATE_PCIE_M1_AUX_CLK, "gate_pcie_m1_aux", "fix_pll3_396", 0x80, 17},
        {CXD_CLK_GATE_PCIE_C_PCLK, "gate_pcie_c_p", "fix_pll3_99", 0x80, 19},
        {CXD_CLK_GATE_PCIE_C_DBI_CLK, "gate_pcie_c_dbi", "fix_pll3_396", 0x80, 20},
        {CXD_CLK_GATE_PCIE_C_SLV_CLK, "gate_pcie_c_slv", "fix_pll3_396", 0x80, 21},
        {CXD_CLK_GATE_PCIE_C_MST_CLK, "gate_pcie_c_mst", "fix_pll3_396", 0x80, 22},
        {CXD_CLK_GATE_PCIE_C_AUX_CLK, "gate_pcie_c_aux", "fix_pll4_1", 0x80, 23},
        {CXD_CLK_GATE_PCIE_N_PCLK, "gate_pcie_n_p", "fix_pll3_99", 0x80, 25},
        {CXD_CLK_GATE_PCIE_N_DBI_CLK, "gate_pcie_n_dbi", "fix_pll3_396", 0x80, 26},
        {CXD_CLK_GATE_PCIE_N_SLV_CLK, "gate_pcie_n_slv", "fix_pll3_396", 0x80, 27},
        {CXD_CLK_GATE_PCIE_N_MST_CLK,	"gate_pcie_n_mst", "fix_pll3_396", 0x80, 28},
        {CXD_CLK_GATE_PCIE_N_AUX_CLK, "gate_pcie_n_aux", "fix_pll4_1", 0x80, 29},
};

#ifdef CONFIG_DEBUG_FS
static struct debugfs_reg32 cxd900xx_clk_debug_reg[] = {
        {"0x0000", 0x0000},
        {"0x0010", 0x0010},
        {"0x0020", 0x0020},
        {"0x0030", 0x0030},
        {"0x0040", 0x0040},
        {"0x0050", 0x0050},
        {"0x0060", 0x0060},
        {"0x0070", 0x0070},
        {"0x0080", 0x0080},
        {"0x0100", 0x0100},
        {"0x0104", 0x0104},
        {"0x0108", 0x0108},
        {"0x010C", 0x010C},
        {"0x0110", 0x0110},
        {"0x0114", 0x0114},
        {"0x0118", 0x0118},
        {"0x011C", 0x011C},
        {"0x0120", 0x0120},
        {"0x0124", 0x0124},
        {"0x0128", 0x0128},
        {"0x0200", 0x0200},
        {"0x0210", 0x0210},
        {"0x0220", 0x0220},
        {"0x0230", 0x0230},
        {"0x0240", 0x0240},
};
#endif

#ifdef CONFIG_PM
#include <linux/syscore_ops.h>

struct clk_save {
	int offset;
	u32 mask;
	u32 data;
};
static struct clk_save clksave[] = {
	{ 0x210, 0x1F000000, },
	{ 0x230, 0x000003FF, },
};
#define N_CLKSAVE ARRAY_SIZE(clksave)

static int cxd900xx_clks_suspend(void)
{
	int i;

	for (i = 0; i < N_CLKSAVE; i++) {
		regmap_read(cxd_clk.clk_regmap, clksave[i].offset, &clksave[i].data);
	}

	if (debug) {
		/* show counts */
		for (i = 0; i < cxd_clk.clk_num; i++) {
			unsigned int enable_count;

			if (!cxd_clk.clk[i])
				continue;
			enable_count = __clk_get_enable_count(cxd_clk.clk[i]);
			if (enable_count) {
				printk(KERN_ERR "CLK%s: %u\n", __clk_get_name(cxd_clk.clk[i]), enable_count);
			}
		}
	}
	return 0;
}

static void cxd900xx_clks_resume(void)
{
	int i;
	u32 clr, set;

	for (i = 0; i < N_CLKSAVE; i++) {
		clr = ~clksave[i].data & clksave[i].mask;
		set =  clksave[i].data & clksave[i].mask;
		/* CLR then SET */
		regmap_write(cxd_clk.clk_regmap, CLK_CLR_REG(clksave[i].offset), clr);
		regmap_write(cxd_clk.clk_regmap, CLK_SET_REG(clksave[i].offset), set);
	}
}

static struct syscore_ops cxd900xx_clks_syscore_ops = {
	.suspend = cxd900xx_clks_suspend,
	.resume  = cxd900xx_clks_resume,
};
#endif /* CONFIG_PM */

static void __init cxd_900xx_clks_init(struct device_node *node)
{
        int ret;

        cxd_clk.clk_regmap = syscon_regmap_lookup_by_phandle(node, "syscon");
        if (IS_ERR(cxd_clk.clk_regmap)) {
                err_print("syscon regmap error\n");
                goto out;
        }

        cxd_clk.clk_num = CXD_CLK_END;

        cxd_clk.clk = kzalloc((cxd_clk.clk_num * sizeof(struct clk*)), GFP_KERNEL);
        if (IS_ERR(cxd_clk.clk)) {
                err_print("clk memoory allocate fail\n");
                goto out;
        }

        spin_lock_init(&cxd_clk.lock);

        cxd_clk_fix_set(&cxd_clk, cxd900xx_clk_fix, ARRAY_SIZE(cxd900xx_clk_fix));

        cxd_clk_div_set(&cxd_clk, cxd900xx_clk_div, ARRAY_SIZE(cxd900xx_clk_div));

        cxd_clk_mux_set(&cxd_clk, cxd900xx_clk_mux, ARRAY_SIZE(cxd900xx_clk_mux));

        cxd_clk_gate_set(&cxd_clk, cxd900xx_clk_gate, ARRAY_SIZE(cxd900xx_clk_gate));

        cxd_clk.clk_data.clks = cxd_clk.clk;

        cxd_clk.clk_data.clk_num = cxd_clk.clk_num;

        ret = of_clk_add_provider(node, of_clk_src_onecell_get, &cxd_clk.clk_data);
        if (ret) {
                err_print("could not register clock provider: %d\n", ret);
                goto out;
        }

#ifdef CONFIG_PM
	register_syscore_ops(&cxd900xx_clks_syscore_ops);
#endif

#ifdef CONFIG_DEBUG_FS
        cxd_clk_debugfs_regset_set(cxd_clk.clk_regmap, cxd900xx_clk_debug_reg,
                        ARRAY_SIZE(cxd900xx_clk_debug_reg));
#endif

        printk("CXD clock driver init done\n");

out:
        return;

}
CLK_OF_DECLARE(cxd_900xx, "cxd,900xx-clock", cxd_900xx_clks_init);

