/*
 * Copyright 2018 Sony Imaging Products & Solutions Inc.
 * Copyright 2018 Sony Corporation.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * Multiplixer clock implementation
 */

#include <linux/clk-provider.h>
#include <linux/regmap.h>
#include <linux/slab.h>

#include "cxd-clk-define.h"
#include "cxd-clk-mux.h"

#define to_cxd_clk_mux(_hw) container_of(_hw, struct cxd_clk_mux, hw)

static int clk_mux_val_to_index(struct clk_hw *hw, u32 *table, unsigned int flags,
                unsigned int val)
{
        int num_parents = clk_hw_get_num_parents(hw);

        if (table) {
                int i;

                for (i = 0; i < num_parents; i++)
                        if (table[i] == val)
                                return i;
                return -EINVAL;
        }

        if (val && (flags & CLK_MUX_INDEX_BIT))
                val = ffs(val) - 1;

        if (val && (flags & CLK_MUX_INDEX_ONE))
                val--;

        if (val >= num_parents)
                return -EINVAL;

        return val;
}

static unsigned int clk_mux_index_to_val(u32 *table, unsigned int flags, u8 index)
{
        unsigned int val = index;

        if (table) {
                val = table[index];
        } else {
                if (flags & CLK_MUX_INDEX_BIT)
                        val = 1 << index;

                if (flags & CLK_MUX_INDEX_ONE)
                        val++;
        }

        return val;
}

static void _cxd_clk_mux_set_parent(struct cxd_clk_mux *mux, u32 val);
static int cxd_clk_mux_set_parent(struct clk_hw *hw, u8 index)
{
        struct cxd_clk_mux *mux = to_cxd_clk_mux(hw);
        unsigned long flags = 0;
        u32 val;

        val = clk_mux_index_to_val(NULL, mux->flags, index);

        if (mux->lock)
	{
                spin_lock_irqsave(mux->lock, flags);
		_cxd_clk_mux_set_parent(mux, val);
                spin_unlock_irqrestore(mux->lock, flags);
	} else {
		_cxd_clk_mux_set_parent(mux, val);
	}

        return 0;
}

static void _cxd_clk_mux_set_parent(struct cxd_clk_mux *mux, u32 val)
{
        u32 reg_val;

        reg_val = mux->mask << mux->shift;
        regmap_write(mux->regmap, mux->clr_reg, reg_val);

        reg_val = (val << mux->shift);
        regmap_write(mux->regmap, mux->set_reg, reg_val);

#ifdef CONFIG_ARCH_CXD900XX_FPGA
        regmap_read(mux->regmap, mux->sta_reg, &reg_val);
        reg_val &= ~(mux->mask << mux->shift);
        reg_val |= (val << mux->shift);
        regmap_write(mux->regmap, mux->sta_reg, reg_val);
#endif
}

static u8 cxd_clk_mux_get_parent(struct clk_hw *hw)
{
        struct cxd_clk_mux *mux = to_cxd_clk_mux(hw);
        u32 reg_val = 0;

        regmap_read(mux->regmap, mux->sta_reg, &reg_val);
        reg_val = reg_val >> mux->shift;
        reg_val &= mux->mask;

        return clk_mux_val_to_index(hw, NULL, mux->flags, reg_val);
}

static struct clk_ops cxd_clk_mux_ops = {
        .set_parent = cxd_clk_mux_set_parent,
        .get_parent = cxd_clk_mux_get_parent,
};

struct clk *cxd_clk_register_mux(struct device *dev, const char *name,
                const char * const *parent_names, u32 num_parent, unsigned long flags,
                struct regmap *regmap, u32 sta_ofs, u32 set_ofs, u32 clr_ofs,
                u8 bit_ofs, u8 bit_width, u8 clk_gate_flags, spinlock_t *lock)
{
        struct cxd_clk_mux *mux;
        struct clk *clk;
        struct clk_init_data init;

        mux = kzalloc(sizeof(struct cxd_clk_mux), GFP_KERNEL);
        if (!mux) {
                err_print("memory allocate fail\n");
                return ERR_PTR(-ENOMEM);
        }

        /* struct cxd_clk_gate assignments */
        mux->regmap = regmap;
        mux->sta_reg = sta_ofs;
        mux->set_reg = set_ofs;
        mux->clr_reg = clr_ofs;
        mux->mask = BIT(bit_width) - 1;
        mux->shift = bit_ofs;
        mux->width = bit_width;
        mux->flags = flags;
        mux->lock = lock;

        init.name = name;
        init.ops = &cxd_clk_mux_ops;
        init.flags = flags;
        init.parent_names = parent_names ? parent_names : NULL;
        init.num_parents =  parent_names ? num_parent : 0;

        mux->hw.init = &init;

        clk = clk_register(dev, &mux->hw);
        if (IS_ERR(clk)) {
                err_print("register mux clock error\n");
                kfree(mux);
        }

        return clk;
}

