

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/extcon.h>
#include <linux/semaphore.h>
#include <linux/kthread.h>
#include <linux/usb/f_usb/fusb_dwc3.h>
#include "core.h"
#include "dwc3-of-cxd.h"
#include "cxd-phy.h"

#include "crphy_defs.h"

/* TCA Register Map*/
#define TCA_INTR_EN             0x4
#define TCA_INTR_STS            0x8
#define TCA_GCFG                0x10
#define TCA_TCPC                0x14
#define TCA_CTRLSYNCMODE_CFG0   0x20
#define TCA_GEN_STATUS          0x34

#define TCA_INTR_STS_MASK       0xffffffff
#define TCA_INTR_STS_ACK_EVT    (1<<0)

#define TCA_FLD_ACK_EN          (1<<0)
#define TCA_FLD_TIMEOUT_EN      (1<<1)
#define TCA_GCFG_FLD_OP_SYSMOD_MASK  (3<<0)
#define TCA_GCFG_FLD_OP_SYSMOD  (1<<0)
#define TCA_GCFG_FLD_ROLE_HSTDEV  (1<<4)
#define TCA_GEN_PHY_DISABLE     (1<<3)

#define TCA_TCPC_MUX_MASK       (0x3)
#define TCA_TCPC_MUX_CTL_USB31  (1<<0)
#define TCA_TCPC_CONNECTOR_ORIENTATION  (1<<2)
#define TCA_TCPC_LOW_POWER_EN   (1<<3)
#define TCA_TCPC_VALID          (1<<4)

#define TCA_CTRLSYNCMODE_CFG0_BLOCK_SS_OP (1<<0)

/* address offset to phy */
#define TCA_OFFSET_TOP          (DWC3_PHY_OFFSET - DWC3_GLOBALS_REGS_START)

int teardown_cxd_uphy(struct dwc3 *dwc)
{
	void __iomem *phy_addr = dwc->regs + TCA_OFFSET_TOP;
	u32	value;

	//Disable TCA IRQ
	value = readl(phy_addr + TCA_INTR_EN) & ~(TCA_FLD_ACK_EN | TCA_FLD_TIMEOUT_EN);
	writel(value, (phy_addr + TCA_INTR_EN));


	//Clear TCA IRQ
	value = readl(phy_addr + TCA_INTR_STS) | TCA_INTR_STS_MASK;
	writel(value, (phy_addr + TCA_INTR_STS));
	return 0;
}

int setup_cxd_uphy(struct dwc3 *dwc)
{
	void __iomem *phy_addr = dwc->regs + TCA_OFFSET_TOP;
	u32	value;
	u32	retries=1000;

	//Clear TCA IRQ
	value = readl(phy_addr + TCA_INTR_STS) | TCA_INTR_STS_MASK;
	writel(value, (phy_addr + TCA_INTR_STS));

	//Enable TCA IRQ
	value = readl(phy_addr + TCA_INTR_EN) | (TCA_FLD_ACK_EN | TCA_FLD_TIMEOUT_EN);
	writel(value, (phy_addr + TCA_INTR_EN));

	//set host or device mode
	//value = readl(phy_addr + TCA_GCFG) | TCA_GCFG_FLD_ROLE_HSTDEV;
	//writel(value, (phy_addr + TCA_GCFG));

	// WorkAround: 2020/06/02
	// If not wait 100ms, the USB connection will fail.
	// wait time are under investigation.
	mdelay(100);

	//Setup TCA_TCPC register
	value = readl(phy_addr + TCA_TCPC);
	//MUX control
	value &= ~(TCA_TCPC_MUX_MASK);
	value |= TCA_TCPC_MUX_CTL_USB31;
	value &= ~(TCA_TCPC_CONNECTOR_ORIENTATION);
	value &= ~(TCA_TCPC_LOW_POWER_EN);
	value |= TCA_TCPC_VALID;
	if (dwc->desired_typec_orientation)
		value |= TCA_TCPC_CONNECTOR_ORIENTATION;
	writel(value, (phy_addr + TCA_TCPC));
	dwc->current_typec_orientation = dwc->desired_typec_orientation;

	//Polling TCA_INTR_STS_ACK_EVT
	do{
		value = readl(phy_addr + TCA_INTR_STS);
		if((value & TCA_INTR_STS_ACK_EVT))
			goto done;
		udelay(100);
	}while(--retries);

	return -ETIMEDOUT;
done:
	
	value = readl(phy_addr + TCA_CTRLSYNCMODE_CFG0);
	value &= ~(TCA_CTRLSYNCMODE_CFG0_BLOCK_SS_OP);
	if (!dwc->desired_ss_operation)
		value |= TCA_CTRLSYNCMODE_CFG0_BLOCK_SS_OP;
	writel(value, (phy_addr + TCA_CTRLSYNCMODE_CFG0));
	dwc->current_ss_operation = dwc->desired_ss_operation;

	return 0;
}

static int dwc3_cxd_uphy_set_typec_orientation(struct dwc3 *dwc)
{
	void __iomem *phy_addr = dwc->regs + TCA_OFFSET_TOP;
	u32	value;

	if (dwc->current_typec_orientation == dwc->desired_typec_orientation)
		return 0;

	value = readl(phy_addr + TCA_TCPC);
	value &= ~(TCA_TCPC_CONNECTOR_ORIENTATION);
	if (dwc->desired_typec_orientation)
		value |= TCA_TCPC_CONNECTOR_ORIENTATION;

	value |= TCA_TCPC_VALID;
	writel(value, (phy_addr + TCA_TCPC));
	dwc->current_typec_orientation = dwc->desired_typec_orientation;

	return 0;
}

static int dwc3_cxd_uphy_set_ss_operation(struct dwc3 *dwc)
{
	void __iomem *phy_addr = dwc->regs + TCA_OFFSET_TOP;
	u32	value;

	if (dwc->current_ss_operation == dwc->desired_ss_operation)
		return 0;

	value = readl(phy_addr + TCA_CTRLSYNCMODE_CFG0);
	value &= ~(TCA_CTRLSYNCMODE_CFG0_BLOCK_SS_OP);
	if (!dwc->desired_ss_operation)
		value |= TCA_CTRLSYNCMODE_CFG0_BLOCK_SS_OP;

	writel(value, (phy_addr + TCA_CTRLSYNCMODE_CFG0));
	dwc->current_ss_operation = dwc->desired_ss_operation;

	return 0;
}

DEFINE_SEMAPHORE(dwc3_typec_setting_sem);
static int dwc3_typec_notify_exit = 0;
static int dwc3_typec_notification_handler(void *p)
{
	int ret;
	union extcon_property_value polarity_val;
	union extcon_property_value usb_ss_val;
	struct dwc3 *dwc = (struct dwc3 *)p;

	while (1) {
		set_current_state(TASK_INTERRUPTIBLE);
		ret = down_interruptible(&dwc3_typec_setting_sem);
		if (ret) {
			dev_err(dwc->dev, "down err=%d\n", ret);
			BUG_ON(1);
		}
		if (dwc3_typec_notify_exit) break;

		ret = extcon_get_property(dwc->edev, EXTCON_USB,
		          EXTCON_PROP_USB_TYPEC_POLARITY,
		          (union extcon_property_value *)&polarity_val);
		if (ret) {
			dev_err(dwc->dev, "couldn't get EXTCON_PROP_USB_TYPEC_POLARITY, err=%d\n", ret);
			continue;
		}
		ret = extcon_get_property(dwc->edev, EXTCON_USB,
		          EXTCON_PROP_USB_SS,
		          (union extcon_property_value *)&usb_ss_val);
		if (ret) {
			dev_err(dwc->dev, "couldn't get EXTCON_PROP_USB_SS, err=%d\n", ret);
			continue;
		}

		mutex_lock(&dwc->phyadj_mtx);
		dwc->desired_typec_orientation = !!polarity_val.intval;
		dwc3_cxd_uphy_set_typec_orientation(dwc);
		dwc->desired_ss_operation = !!usb_ss_val.intval;
		dwc3_cxd_uphy_set_ss_operation(dwc);
		mutex_unlock(&dwc->phyadj_mtx);
	}
	return 0;
}

static int dwc3_typec_notifier(struct notifier_block *nb,
			     unsigned long event, void *ptr)
{
	printk(KERN_INFO "==>%s(), event = %ld \n", __FUNCTION__, event);
	up(&dwc3_typec_setting_sem);

	return NOTIFY_DONE;
}

int dwc3_cxd_init_typec(struct dwc3 *dwc)
{
	int ret;
	union extcon_property_value prop_val;
	struct notifier_block *nb = &dwc->edev_nb[DWC3_EDEV_TYPE_C];

	if (dwc->edev) {
		ret = extcon_get_property_capability(dwc->edev, EXTCON_USB,
		                                     EXTCON_PROP_USB_TYPEC_POLARITY);
		dev_info(dwc->dev, "%s polarity cap=%d\n", __FUNCTION__, ret);

		prop_val.intval = dwc->desired_typec_orientation;

		ret = extcon_set_property_sync(dwc->edev, EXTCON_USB,
		          EXTCON_PROP_USB_TYPEC_POLARITY, prop_val);
		if (ret < 0) dev_err(dwc->dev, "failed to extcon_set_property_sync\n");

		ret = extcon_get_property_capability(dwc->edev, EXTCON_USB,
		                                     EXTCON_PROP_USB_SS);
		dev_info(dwc->dev, "%s ss cap=%d\n", __FUNCTION__, ret);

		prop_val.intval = dwc->desired_ss_operation;

		ret = extcon_set_property_sync(dwc->edev, EXTCON_USB,
		          EXTCON_PROP_USB_SS, prop_val);
		if (ret < 0) dev_err(dwc->dev, "failed to extcon_set_property_sync\n");

		{
			int err = down_interruptible(&dwc3_typec_setting_sem);
			if (err) {
				dev_err(dwc->dev, "down err=%d\n", err);
			}
		}
		kthread_run(dwc3_typec_notification_handler, dwc, "dwc3_typec_setting");

		nb->notifier_call = dwc3_typec_notifier;
		ret = extcon_register_notifier(dwc->edev, EXTCON_USB, nb);
		if (ret < 0) {
			dev_err(dwc->dev, "couldn't register cable notifier\n");
			return ret;
		}
	}
	else {
		dev_err(dwc->dev, "edev is not initialized\n");
		return -ENODEV;
	}

	return 0;
}

void dwc3_cxd_exit_typec(struct dwc3 *dwc)
{
	dwc3_typec_notify_exit = 1;
	up(&dwc3_typec_setting_sem);

	extcon_unregister_notifier(dwc->edev, EXTCON_USB,
				   &dwc->edev_nb[DWC3_EDEV_TYPE_C]);
}
/*
void dump_u31phy(void __iomem *reg)
{
	int i;
	u32 v[4];
	void __iomem *base;

	for (i=0; ; i+=16) {
		if (0x50 <= i) break;
		base = reg + i;
		v[0] = readl(base);
		v[1] = readl(base + 4);
		v[2] = readl(base + 8);
		v[3] = readl(base + 12);
		printk(KERN_INFO "%s %p: %08x %08x %08x %08x\n", __FUNCTION__, base, v[0], v[1], v[2], v[3]);
	}
	base = reg + i;
	v[0] = readl(base);
	v[1] = readl(base + 4);
	v[2] = readl(base + 8);
	printk(KERN_INFO "%s %p: %08x %08x %08x\n", __FUNCTION__, base, v[0], v[1], v[2]);
}
*/

#define U31PHY_CFGR0               0x0020
#define U31PHY_CFGR0_PHY_REF_BIT   (1 << 0)
#define U31PHY_CFGR0_PHY_REF_PAD   (0)
int setup_cxd_u31phy(void)
{
	u32 value;
	void __iomem *phy_addr = dwc3_u31phy_top();

	if (!phy_addr)
		return -ENODEV;

	/*printk(KERN_INFO "%s phy_addr=%p\n", __FUNCTION__, phy_addr+U31PHY_CFGR0);*/
	value = readl(phy_addr + U31PHY_CFGR0) & ~U31PHY_CFGR0_PHY_REF_BIT;
	value |= U31PHY_CFGR0_PHY_REF_PAD;
	writel(value, phy_addr + U31PHY_CFGR0);

	/*dump_u31phy(phy_addr);*/
	return 0;
}
int teardown_cxd_u31phy(void)
{
	return 0;
}

#define U31CTRL_CFGR0			0x0010
#define U31PHY_CFGR0_GEN1_BIT	(1<<3)
#define U31PHY_CFGR0_u3_disable (1<<2)
int force_u31_gen1(void)
{
	u32 value;
	void __iomem *phy_addr = dwc3_u31phy_top();

	if (!phy_addr)
		return -ENODEV;

	 value = readl(phy_addr + U31CTRL_CFGR0) & ~U31PHY_CFGR0_GEN1_BIT;
	 value |= U31PHY_CFGR0_GEN1_BIT;
	 writel(value, phy_addr + U31CTRL_CFGR0);
	 return 0;
}
int force_u31_to_hs(void)
{
	u32 value;
	void __iomem *phy_addr = dwc3_u31phy_top();//0xf1080000

	if (!phy_addr)
		return -ENODEV;

	value = readl(phy_addr + U31CTRL_CFGR0) & ~U31PHY_CFGR0_u3_disable;
	value |= U31PHY_CFGR0_u3_disable;
	writel(value, phy_addr + U31CTRL_CFGR0);

	return 0;
}


#define U31PHY_CFGR0_SRAM_BYPASS_BIT        (1<<16)
#define U31PHY_CFGR0_SRAM_BYPASS_VAL        (0)
#define U31PHY_CFGR0_SRAM_INIT_DONE_BIT     (1<<18)
#define U31PHY_CFGR0_SRAM_INIT_DONE         U31PHY_CFGR0_SRAM_INIT_DONE_BIT
#define U31PHY_CFGR0_PIPE_LANE0_RESET_BIT   (1<<19)
#define U31PHY_CFGR0_PIPE_LANE0_RESET_DEASSERT_VAL   U31PHY_CFGR0_PIPE_LANE0_RESET_BIT
#define U31PHY_CFGR0_PIPE_LANE0_RESET_ASSERT_VAL     (0)
#define U31PHY_CFGR0_CR_PARA_SEL_BIT        (1<<20)
#define U31PHY_CFGR0_CR_PARA_SEL_VAL        U31PHY_CFGR0_CR_PARA_SEL_BIT
static int cr_para_sel_enabled = 0;
int enable_cr_para_sel(void __iomem *reg)
{
	u32 value;

	if (!reg)
		return -ENODEV;

	value = readl(reg + U31PHY_CFGR0) & ~U31PHY_CFGR0_SRAM_BYPASS_BIT;
	value |= U31PHY_CFGR0_SRAM_BYPASS_VAL;
	writel(value, reg + U31PHY_CFGR0);

	value = readl(reg + U31PHY_CFGR0) & ~U31PHY_CFGR0_CR_PARA_SEL_BIT;
	value |= U31PHY_CFGR0_CR_PARA_SEL_VAL;
	printk(KERN_INFO "%s write U31PHY_CFGR0 - PARA SEL\n", __FUNCTION__);
	writel(value, reg + U31PHY_CFGR0);
	cr_para_sel_enabled = 1;

	return 0;
}

int disable_cr_para_sel(void __iomem *reg)
{
	u32 value;

	if (!reg)
		return -ENODEV;

	cr_para_sel_enabled = 0;
	value = readl(reg + U31PHY_CFGR0) & ~U31PHY_CFGR0_CR_PARA_SEL_BIT;
	writel(value, reg + U31PHY_CFGR0);

	value = readl(reg + U31PHY_CFGR0) & ~U31PHY_CFGR0_SRAM_BYPASS_BIT;
	value |= U31PHY_CFGR0_SRAM_BYPASS_BIT;
	writel(value, reg + U31PHY_CFGR0);

	return 0;
}

int check_cr_sram_init_done(void __iomem *reg)
{
	u32 value;
	u32	retries=1000;

	do {
		mb();
		value = readl(reg + U31PHY_CFGR0) & U31PHY_CFGR0_SRAM_INIT_DONE;
		if (value) goto done;
		udelay(1000);
	} while (--retries);

	printk(KERN_ERR "%s timed out to wait for SRAM_INIT_DONE\n", __FUNCTION__);
	return -ETIMEDOUT;

done:
	printk(KERN_INFO "%s SRAM_INIT_DONE\n", __FUNCTION__);
	return 0;
}

#define U31PHY_CFGR0_SRAM_EXT_LD_DONE_BIT   (1<<17)
#define U31PHY_CFGR0_SRAM_EXT_LD_DONE_VAL   U31PHY_CFGR0_SRAM_EXT_LD_DONE_BIT
int cr_para_sel_done(void __iomem *reg)
{
	u32 value;

	if (!reg)
		return -ENODEV;

	value = readl(reg + U31PHY_CFGR0) & ~U31PHY_CFGR0_SRAM_EXT_LD_DONE_BIT;
	value |= U31PHY_CFGR0_SRAM_EXT_LD_DONE_VAL;
	writel(value, reg + U31PHY_CFGR0);

	printk(KERN_INFO "%s SRAM_EXT_LD_DONE\n", __FUNCTION__);
	return 0;
}

#define clrmsk(fld)     ~(fld##_MSK << fld##_MAG)
#define adjval(p, fld)  (((uint32_t)p[fld##_BOFS] & fld##_MSK) << fld##_MAG)
int u2phy_adjust(void __iomem *reg, const void *_data,  const void *_port)
{
	uint32_t value;
	int ofs = 0;
	const uint8_t *prm;
	const uint8_t *data = (const uint8_t*)_data;
	const int *port = (const int *)_port;

	printk(KERN_INFO "%s usb:%s\n", __FUNCTION__, ((*port == FUSB_DWC3_PORT_TYPE_MICRO) ? "micro" :
	                                               (*port == FUSB_DWC3_PORT_TYPE_TYPEC) ? "typec" : "unknown"));

	if (*port == FUSB_DWC3_PORT_TYPE_TYPEC)
		ofs = U2PHY_TPC_DATA_START_BOFS;
	prm = data + ofs;

	value = readl(reg + U2PHY_CFGR0);

	value &= clrmsk(U2PHY_CFGR0_r_COMPDISTUNE0);
	value |= adjval(prm, U2PHY_CFGR0_r_COMPDISTUNE0);

	writel(value, reg + U2PHY_CFGR0);


	value = readl(reg + U2PHY_CFGR1);

	value &= clrmsk(U2PHY_CFGR1_r_TXVREFTUNE0);
	value |= adjval(prm, U2PHY_CFGR1_r_TXVREFTUNE0);

	value &= clrmsk(U2PHY_CFGR1_r_TXPREEMPPULSETUNE0);
	value |= adjval(prm, U2PHY_CFGR1_r_TXPREEMPPULSETUNE0);

	value &= clrmsk(U2PHY_CFGR1_r_TXPREEMPAMPTUNE0);
	value |= adjval(prm, U2PHY_CFGR1_r_TXPREEMPAMPTUNE0);

	writel(value, reg + U2PHY_CFGR1);


	value = readl(reg + U2PHY_CFGR0);

	value &= clrmsk(U2PHY_CFGR0_r_SQRXTUNE0);
	value |= adjval(prm, U2PHY_CFGR0_r_SQRXTUNE0);

	writel(value, reg + U2PHY_CFGR0);

	return 0;
}
#define RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_INVALID_MSK  (u32)(~(u16)-1)

static u32
    RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[CRPHY_NUM_LANES];

void cr_phy31_adjust_init(void)
{
	int lane;

	for (lane=0; lane<CRPHY_NUM_LANES; lane++)
		RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[lane]
		    = RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_INVALID_MSK;
}

int cr_phy31_adjust(void __iomem *reg, const void *_data)
{
	u16 value;
	int lane, ofs;
	const uint8_t *prm;
	const uint8_t *data = (const uint8_t *)_data;

	/*
	 *    SS/SSP Tx equalizer tuning
	 */
	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + LANEN_DIG_ASIC_TX_OVRD_IN_1(lane));
		value &= clrmsk(LDA_TX_MAIN_OVRD_EN);
		value |= adjval(prm, LDA_TX_MAIN_OVRD_EN);
		value &= clrmsk(LDA_TX_MAIN_CURSOR);
		value |= adjval(prm, LDA_TX_MAIN_CURSOR);
		writew(value, reg + LANEN_DIG_ASIC_TX_OVRD_IN_1(lane));
	}

	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + LANEN_DIG_ASIC_TX_OVRD_IN_2(lane));
		value &= clrmsk(LDA_TX_POST_OVRD_EN);
		value |= adjval(prm, LDA_TX_POST_OVRD_EN);
		value &= clrmsk(LDA_TX_POST_CURSOR);
		value |= adjval(prm, LDA_TX_POST_CURSOR);
		writew(value, reg + LANEN_DIG_ASIC_TX_OVRD_IN_2(lane));

		/*value = readw(reg + LANEN_DIG_ASIC_TX_OVRD_IN_2(lane));*/
		value &= clrmsk(LDA_TX_PRE_OVRD_EN);
		value |= adjval(prm, LDA_TX_PRE_OVRD_EN);
		value &= clrmsk(LDA_TX_PRE_CURSOR);
		value |= adjval(prm, LDA_TX_PRE_CURSOR);
		writew(value, reg + LANEN_DIG_ASIC_TX_OVRD_IN_2(lane));
	}

	/*
	 *    SS/SSP Tx Amplitude Adjustment
	 */
	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane));
		/* RESET_OVRD should be varied */
		value &= clrmsk(RD_RESET_OVRD_EN);
		value |= adjval(prm, RD_RESET_OVRD_EN);
		value &= clrmsk(RD_RESET_OVRD_VAL);
		value |= adjval(prm, RD_RESET_OVRD_VAL);

		value &= clrmsk(RD_VBOOST_EN_OVRD_EN);
		value |= adjval(prm, RD_VBOOST_EN_OVRD_EN);
		value &= clrmsk(RD_VBOOST_EN_OVRD_VAL);
		value |= adjval(prm, RD_VBOOST_EN_OVRD_VAL);
		writew(value, reg + RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane));
	}

	value = readw(reg + SUP_DIG_LVL_OVRD_IN);   /* this register for all lanes */
	value &= clrmsk(SD_LVL_TX_VBOOST_LVL_EN);
	value |= adjval(data,                       /* take parameter from common */
	                SD_LVL_TX_VBOOST_LVL_EN);
	value &= clrmsk(SD_LVL_TX_VBOOST_LVL);
	value |= adjval(data,                       /* take parameter from common */
	                SD_LVL_TX_VBOOST_LVL);
	writew(value, reg + SUP_DIG_LVL_OVRD_IN);

	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane));
		value &= clrmsk(RD_IBOOST_LVL_OVRD_EN);
		value |= adjval(prm, RD_IBOOST_LVL_OVRD_EN);
		value &= clrmsk(RD_IBOOST_LVL_OVRD_VAL);
		value |= adjval(prm, RD_IBOOST_LVL_OVRD_VAL);
		writew(value, reg + RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane));

		RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[lane] = (u32)value;
	}

	/*
	 *    SS/SSP Loss-of-Signal Detector Threshold Level Control
	 */
	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + LANEN_DIG_ASIC_RX_OVRD_IN_3(lane));
		value &= clrmsk(LDA_LOS_OVRD_EN);
		value |= adjval(prm, LDA_LOS_OVRD_EN);
		value &= clrmsk(LDA_LOS_LFPS_EN);
		value |= adjval(prm, LDA_LOS_LFPS_EN);
		value &= clrmsk(LDA_LOS_THRSHLD);
		value |= adjval(prm, LDA_LOS_THRSHLD);
		value &= clrmsk(LDA_DISABLE_OVRD_EN);
		value |= adjval(prm, LDA_DISABLE_OVRD_EN);
		value &= clrmsk(LDA_DISABLE);
		value |= adjval(prm, LDA_DISABLE);
		writew(value, reg + LANEN_DIG_ASIC_RX_OVRD_IN_3(lane));
	}

	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + RAWLANEN_DIG_RX_CTL_RX_LOS_MASK_CTL(lane));
		value &= clrmsk(RD_RX_LOS_MASK_CNT_LSB);
		value |= adjval(prm, RD_RX_LOS_MASK_CNT_LSB);
		value &= clrmsk(RD_RX_LOS_MASK_CNT_MSB);
		value |= adjval(prm, RD_RX_LOS_MASK_CNT_MSB);
		writew(value, reg + RAWLANEN_DIG_RX_CTL_RX_LOS_MASK_CTL(lane));
	}

	/*
	 *    SS/SSP RX equalizer setting
	 */
	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane));
		value &= clrmsk(LDA_RX_EQ_OVRD_EN);
		value |= adjval(prm, LDA_RX_EQ_OVRD_EN);
		writew(value, reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane));
	}

	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_0(lane));
		value &= clrmsk(LDA_RX_EQ_CTLE_BOOST);
		value |= adjval(prm, LDA_RX_EQ_CTLE_BOOST);
		value &= clrmsk(LDA_RX_EQ_AFE_GAIN);
		value |= adjval(prm, LDA_RX_EQ_AFE_GAIN);
		value &= clrmsk(LDA_RX_EQ_ATT_LVL);
		value |= adjval(prm, LDA_RX_EQ_ATT_LVL);
		writew(value, reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_0(lane));
	}

	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane));
		value &= clrmsk(LDA_RX_EQ_DFE_TAP1);
		value |= adjval(prm, LDA_RX_EQ_DFE_TAP1);
		value &= clrmsk(LDA_RX_EQ_DFE_TAP2);
		value |= adjval(prm, LDA_RX_EQ_DFE_TAP2);
		writew(value, reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane));
	}

	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {
		ofs = CRPHY_DATA_START_BOFS + lane * CRPHY_NUM_LANE_DATA;
		prm = data + ofs;

		value = readw(reg + RAWLANEN_DIG_PCS_XF_RX_EQ_DELTA_IQ_OVRD_IN(lane));
		value &= clrmsk(RD_RX_EQ_DELTA_IQ_OVRD_EN);
		value |= adjval(prm, RD_RX_EQ_DELTA_IQ_OVRD_EN);
		value &= clrmsk(RD_RX_EQ_DELTA_IQ_OVRD_VAL);
		value |= adjval(prm, RD_RX_EQ_DELTA_IQ_OVRD_VAL);
		writew(value, reg + RAWLANEN_DIG_PCS_XF_RX_EQ_DELTA_IQ_OVRD_IN(lane));
	}

	return 0;
}

int cr_phy31_adjust_post_csftrst(void)
{
	u16 value;
	int lane;
	void __iomem *reg = dwc3_crphy_top();

	if (!reg)
		return -ENODEV;

	if (RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[0]
		    & RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_INVALID_MSK) {
		printk(KERN_INFO "%s no adjust value, skip (0x%x, mask=0x%x)\n", __FUNCTION__, RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[0], RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_INVALID_MSK);
		return -ENOENT;
	}

	printk(KERN_INFO "%s post adjust, value[lane0]=0x%x\n", __FUNCTION__, RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[0]);
	for (lane=0; lane<CRPHY_NUM_LANES; lane ++) {

		value = RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[lane];
		writew(value, reg + RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane));

		/* refresh not to use twice */
		RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_post_csftrst[lane]
		    = RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1_INVALID_MSK;
	}
	return 0;
}

#ifdef CONFIG_USB_DWC3_OF_CXD_PHY_ADJUST_DEBUG
static char dwc3_phy_reg_snapshot[2048];
static int dwc3_phy_reg_snapshot_len = 0;
int _snapshot_one(void __iomem *reg, uint32_t ofs, const char *pfx, int bl)
{
	int sz;
	uint32_t value;

	sz = sizeof(dwc3_phy_reg_snapshot) - bl;
	if (0 >= sz) return -1;
	value = readl(reg + ofs);
	return snprintf(&dwc3_phy_reg_snapshot[bl], sz,
	              "%s%04X=0x%08x\n", pfx, ofs, value);
}

void snapshot_phy_param(void __iomem *u2reg, void __iomem *u31reg)
{
	int lane, ret;
	int bl = 0;

	memset(dwc3_phy_reg_snapshot, 0, sizeof(dwc3_phy_reg_snapshot));
	dwc3_phy_reg_snapshot_len = 0;

	/* u2 phy */
	ret = _snapshot_one(u2reg, U2PHY_CFGR0, "0xF108", bl);
	if (0 > ret) goto bail;
	bl += ret;
	ret = _snapshot_one(u2reg, U2PHY_CFGR1, "0xF108", bl);
	if (0 > ret) goto bail;
	bl += ret;

	/* u31phy */
	ret = _snapshot_one(u31reg, SUP_DIG_LVL_OVRD_IN, "0xF10A", bl);   /* this register for all lanes */
	if (0 > ret) goto bail;
	bl += ret;

	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, LANEN_DIG_ASIC_TX_OVRD_IN_1(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, LANEN_DIG_ASIC_TX_OVRD_IN_2(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, LANEN_DIG_ASIC_RX_OVRD_IN_3(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, RAWLANEN_DIG_RX_CTL_RX_LOS_MASK_CTL(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, LANEN_DIG_ASIC_RX_OVRD_EQ_IN_0(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		ret = _snapshot_one(u31reg, RAWLANEN_DIG_PCS_XF_RX_EQ_DELTA_IQ_OVRD_IN(lane), "0xF10A", bl);
		if (0 > ret) goto bail;
		bl += ret;
	}
	dwc3_phy_reg_snapshot_len = bl;
	return;

bail:
	dwc3_phy_reg_snapshot_len = snprintf(dwc3_phy_reg_snapshot,
	                                     sizeof(dwc3_phy_reg_snapshot),
	                                     "(no buffer space)\n");
}

int read_snapshot_phy_param(char *buf)
{
	memcpy(buf, dwc3_phy_reg_snapshot, dwc3_phy_reg_snapshot_len);
	return dwc3_phy_reg_snapshot_len;
}
#endif

/*
void setup_cr_phy_dump(void)
{
	void __iomem *reg;
	int lane;
	u32 value;

	if (!cr_para_sel_enabled) {
		printk(KERN_INFO "%s cr phy is not enabled\n", __FUNCTION__);
		return;
	}

	reg = dwc3_u31phy_top();
	value = readl(reg + U2PHY_CFGR0);
	printk(KERN_INFO "%s u2 ofs=%x, value=0x%x\n", __FUNCTION__, U2PHY_CFGR0, value);
	value = readl(reg + U2PHY_CFGR1);
	printk(KERN_INFO "%s u2 ofs=%x, value=0x%x\n", __FUNCTION__, U2PHY_CFGR1, value);

	reg = dwc3_crphy_top();
	value = readl(reg + SUP_DIG_LVL_OVRD_IN);
	printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, SUP_DIG_LVL_OVRD_IN, value);
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + LANEN_DIG_ASIC_TX_OVRD_IN_1(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, LANEN_DIG_ASIC_TX_OVRD_IN_1(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + LANEN_DIG_ASIC_TX_OVRD_IN_2(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, LANEN_DIG_ASIC_TX_OVRD_IN_2(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, RAWLANEN_DIG_PCS_XF_TX_OVRD_IN_1(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + LANEN_DIG_ASIC_RX_OVRD_IN_3(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, LANEN_DIG_ASIC_RX_OVRD_IN_3(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + RAWLANEN_DIG_RX_CTL_RX_LOS_MASK_CTL(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, RAWLANEN_DIG_RX_CTL_RX_LOS_MASK_CTL(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, LANEN_DIG_ASIC_RX_OVRD_EQ_IN_1(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + LANEN_DIG_ASIC_RX_OVRD_EQ_IN_0(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, LANEN_DIG_ASIC_RX_OVRD_EQ_IN_0(lane), value);
	}
	for (lane=0; lane<CRPHY_NUM_LANES; lane++) {
		value = readl(reg + RAWLANEN_DIG_PCS_XF_RX_EQ_DELTA_IQ_OVRD_IN(lane));
		printk(KERN_INFO "%s u31 ofs=%x, value=0x%x\n", __FUNCTION__, RAWLANEN_DIG_PCS_XF_RX_EQ_DELTA_IQ_OVRD_IN(lane), value);
	}
}
*/

#include "u3_phy_fw_100a_cust1_patch3.h"

void download_snps_u3_phy_fw(void)
{
	void __iomem *phy_cr_addr = dwc3_crphy_top();
	uint16_t *p;
	int i;

	p = u3_phy_fw_100a_cust1_patch3;
	for (i = 0; i < 0x1000; i++) {
		writel(*p++, phy_cr_addr + (0x6000 + i) * 4);
	}
}
