/**
 * drd.c - DesignWare USB3 DRD Controller Dual-role support
 *
 * Copyright (C) 2017 Texas Instruments Incorporated - http://www.ti.com
 *
 * Authors: Roger Quadros <rogerq@ti.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2  of
 * the License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/extcon.h>

#include "debug.h"
#include "core.h"
#include "gadget.h"

static void dwc3_drd_update(struct dwc3 *dwc)
{
	int id;

	id = extcon_get_state(dwc->edev, EXTCON_USB_HOST);
	if (id < 0)
		id = 0;

	dwc3_set_mode(dwc, id ?
		      DWC3_GCTL_PRTCAP_HOST :
		      DWC3_GCTL_PRTCAP_DEVICE);
}

static int dwc3_drd_notifier(struct notifier_block *nb,
			     unsigned long event, void *ptr)
{
	struct dwc3 *dwc = container_of(nb, struct dwc3, edev_nb[DWC3_EDEV_USB_HOST]);
	printk("==>dwc3_drd_notifier(), event = %ld \n", event);
	dwc3_set_mode(dwc, event ?
		      DWC3_GCTL_PRTCAP_HOST :
		      DWC3_GCTL_PRTCAP_DEVICE);

	return NOTIFY_DONE;
}

int dwc3_drd_init(struct dwc3 *dwc)
{
	int ret;
	struct notifier_block *nb = &dwc->edev_nb[DWC3_EDEV_USB_HOST];

	if (dwc->edev) {
		nb->notifier_call = dwc3_drd_notifier;
		ret = extcon_register_notifier(dwc->edev, EXTCON_USB_HOST, nb);
		if (ret < 0) {
			dev_err(dwc->dev, "couldn't register cable notifier err=%d\n", ret);
			return ret;
		}
		dwc3_drd_update(dwc);
	}
	else {
		dev_err(dwc->dev, "edev is not initialized\n");
		return -ENODEV;
	}

	return 0;
}

void dwc3_drd_exit(struct dwc3 *dwc)
{
	extcon_unregister_notifier(dwc->edev,
				   EXTCON_USB_HOST,
				   &dwc->edev_nb[DWC3_EDEV_USB_HOST]);

	cancel_work_sync(&dwc->drd_work);

	switch (dwc->current_dr_role) {
	case DWC3_GCTL_PRTCAP_HOST:
		dwc3_host_exit(dwc);
		break;
	case DWC3_GCTL_PRTCAP_DEVICE:
		dwc3_gadget_exit(dwc);
		break;
	default:
		break;
	}
}
