/**
 *	@file	usbg_iap_pvt.h
 *	@brief	USB iAP(iAP Device Interface) - private header
 *	
 *	Copyright 2024 Sony Group Corporation
 */

#ifndef __USBG_IAP_PVT_H__
#define __USBG_IAP_PVT_H__

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <asm/page.h>

#include <linux/udif/spinlock.h>
#include <linux/udif/cdev.h>

#include "usbg_iap_conf.h"

#ifdef  IAP_DEBUG_ERR
#define IAP_ERR(fmt, args...) { printk("%s(%d): [E]" fmt, FNAME, __LINE__, ##args); }
#else
#define IAP_ERR(fmt, args...) { }
#endif

#ifdef  IAP_DEBUG_INF
#define IAP_INF(fmt, args...) { printk("%s(%d): [I]" fmt, FNAME, __LINE__, ##args);   }
#else
#define IAP_INF(fmt, args...) { }
#endif

#ifdef  IAP_DEBUG_API
#define IAP_API()             { printk("%s(%d): [A]%s()\n", FNAME, __LINE__, __FUNCTION__); }
#else
#define IAP_API()             { }
#endif

#ifdef  IAP_DEBUG_FNC
#define IAP_FNC()             { printk("%s(%d): [F]%s()\n", FNAME, __LINE__, __FUNCTION__); }
#else
#define IAP_FNC()             { }
#endif

#define IAP_LOCK_INI()             { udif_spin_lock_init(   &(iap_get_this_ex()->ep_lock)           ); \
                                     udif_spin_lock_init(   &(iap_get_this_ex()->requesting_lock)   ); }

#define IAP_EP_LOCK_ON()           { udif_spin_lock_irqsave(    &(iap_get_this_ex()->ep_lock), iap_get_this_ex()->ep_lock_flags ); \
                                    /* IAP_INF("### EP LOCK ON  ---> \n"); */ }
#define IAP_EP_LOCK_OFF()          { udif_spin_unlock_irqrestore( &(iap_get_this_ex()->ep_lock), iap_get_this_ex()->ep_lock_flags ); \
                                    /* IAP_INF("### EP LOCK OFF <--- \n"); */ }

#define IAP_REQUESTING_LOCK_ON()   { udif_spin_lock_irqsave(    &(iap_get_this_ex()->requesting_lock), iap_get_this_ex()->requesting_lock_flags ); \
                                    /* IAP_INF("### REQUESTING LOCK ON  ---> \n"); */ }
#define IAP_REQUESTING_LOCK_OFF()  { udif_spin_unlock_irqrestore( &(iap_get_this_ex()->requesting_lock), iap_get_this_ex()->requesting_lock_flags ); \
                                    /* IAP_INF("### REQUESTING LOCK OFF <--- \n"); */ }

#define IAP_FUNC_NAME			"IAP Device"
#define IAP_EP_NAME_BULK_IN 	"IAP data send (Bulk-in) "
#define IAP_EP_NAME_BULK_OUT	"IAP data recv (Bulk-out)"

#define IAP_DAT_BUFSIZE					1024	///< Buffer size (Data)

struct iap_data_req {
    u8 requesting;
    struct usb_request *req;
};

/*
 *	@brief	IAP ep information
 */
struct iap_ep_info {
	u8 ep_no;								///< EP no.
	u8 stalled;								///< stall? or not?
	struct iap_data_req data;				///< usb request(for BULK_IN, BULK_OUT ep)
};

/*
 *	@brief	IAP ep ID
 */
enum {
	IAP_EP_IDX_BULK_IN = 0,					///< ID: Bulk-in  EP
	IAP_EP_IDX_BULK_OUT,				///< ID: Bulk-out EP
	IAP_EP_IDX_MAX,						///< ID: MAX
};

/*
 *	@brief	IAP driver information (extension)
 */
struct func_data_ex {
	bool			activated;				///< driver activated flag
	void*			in_buffer;					///< kernel-virtual address
	void*			out_buffer;					///< kernel-virtual address
	unsigned int	out_used_size;					///< out_buffer used size
	struct iap_ep_info ep_info[IAP_EP_IDX_MAX];	///< EP information
	UDIF_SPINLOCK	ep_lock;							///< lock for endpoint
	unsigned long	ep_lock_flags;
	UDIF_SPINLOCK	requesting_lock;					///< lock for requesting flag
	unsigned long	requesting_lock_flags;
	bool req_comp_waiting;						///< ep request comp wait flag
	struct completion req_wait_completion;		///< ep request comp completion
};

//----------------------------------------------------------------------
// iap_core file operations
UDIF_ERR iap_core_open(UDIF_FILE *filp);
UDIF_ERR iap_core_release(UDIF_FILE *filp);
UDIF_ERR iap_core_ioctl(UDIF_FILE *filp, UDIF_IOCTL *param);
UDIF_SSIZE iap_core_write(UDIF_FILE *filp, UDIF_CDEV_WRITE *param);
UDIF_SSIZE iap_core_read(UDIF_FILE *filp, UDIF_CDEV_READ *param);

//----------------------------------------------------------------------
// iap local functions
struct func_data*		iap_get_this(void);
struct func_data_ex*	iap_get_this_ex(void);
struct usb_ep*			iap_get_ep(int ep_id);
struct iap_ep_info*	iap_get_ep_info(struct usb_ep *ep);

int  iap_create_this(void);
void iap_delete_this(void);
void iap_link_this_to_file(struct file *file);
void iap_request_start(int ep_id);
void iap_request_stop(int ep_id);
int  iap_is_requesting(int ep_id);
void iap_set_requesting_flag( int ep_id, int flag );
void iap_ep_dequeue(struct usb_ep *ep);
void iap_ep_dequeue_all(void);
void iap_ep_dequeue_bulk(void);
void iap_dump(u8 *p, int size, char *fname, int line);
struct usb_request *iap_get_request(struct usb_ep *ep);
bool iap_check_req_comp_waiting(void);
void iap_recheck_req_comp_waiting(void);

#endif
