/*
 * arch/arm/mach-cxd900x0/wdt.c
 *
 * Watchdog timer
 *
 * Copyright 2015,2016,2017,2018 Sony Corporation
 *
 * This code is based on kernel/softlockup.c
 */
/*
 * Detect Soft Lockups
 *
 * started by Ingo Molnar, Copyright (C) 2005, 2006 Red Hat, Inc.
 *
 * this code detects soft lockups: incidents in where on a CPU
 * the kernel does not reschedule for 10 seconds or more.
 */
#include <linux/mm.h>
#include <linux/cpu.h>
#include <linux/nmi.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/freezer.h>
#include <linux/kthread.h>
#include <linux/lockdep.h>
#include <linux/notifier.h>
#include <linux/module.h>
#include <linux/sysctl.h>
#include <linux/interrupt.h>
#include <asm/io.h>
#include <mach/hardware.h>
#include <asm/hardware/arm_scu.h>
#include <asm/hardware/arm_twd.h>
#include <mach/regs-timer.h>
#include <mach/regs-misc.h>
#include <mach/pm_export.h>
#include <linux/snsc_boot_time.h>

#include <mach/errind.h>
#include <linux/udif/module.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/spinlock.h>

/* WDT mode */
#define WDT_OFF  0
#define WDT_ON   1
static unsigned int wdt_mode = WDT_OFF;
module_param_named(mode, wdt_mode, uint, S_IRUSR);
static unsigned int wdt_timeout = 0;
module_param_named(timeout, wdt_timeout, uint, S_IRUSR);

/* wdt time extend to dump in msec */
#define WDT_EXTEDED_TIME_EMDUMP         (2000)
static unsigned int wdt_timeout_mp_excpt = WDT_EXTEDED_TIME_EMDUMP;
module_param_named(mp_excpt, wdt_timeout_mp_excpt, uint, S_IRUSR);

static volatile int s_except = 0;

/*---------- TIMER IP ----------*/
/* WD clock = 1MHz */
#define CXD900X0_WDT_CFG (TMCK_DIV4|TMCS_ONESHOT|TMRST)
static volatile unsigned int cxd900x0_wdt_count; /* TIMER IP reload value */

static void cxd900x0_wdt_setup(void)
{
	writel_relaxed(CXD900X0_WDT_CFG, VA_WDT_TIMER+CXD90014_TIMERCTL);
	writel_relaxed(cxd900x0_wdt_count, VA_WDT_TIMER+CXD90014_TIMERCMP);
}

static void cxd900x0_wdt_start(void)
{
	writel_relaxed(CXD900X0_WDT_CFG|TMST, VA_WDT_TIMER+CXD90014_TIMERCTL);
}

static void cxd900x0_wdt_stop(void)
{
	writel_relaxed(0, VA_WDT_TIMER+CXD90014_TIMERCTL);
	writel_relaxed(TMCLR|TMINTCLR, VA_WDT_TIMER+CXD90014_TIMERCLR);
}

static inline void cxd900x0_wdt_restart(void)
{
	writel_relaxed(TMCLR, VA_WDT_TIMER+CXD90014_TIMERCLR);
}

/*---------- MPCore WDT ----------*/
static unsigned int cpu_wdt_prescaler;
static volatile unsigned int cpu_wdt_count;

#define WDT_CTRL (((cpu_wdt_prescaler-1)<<TWD_TIMER_CONTROL_PRESCALER_SHIFT)|TWD_WDOG_CONTROL_WDMODE)

static void cpu_wdt_setup(void)
{
	writel_relaxed(cpu_wdt_count, VA_LOCALTIMER + TWD_WDOG_COUNTER);
}
static void cpu_wdt_start(void)
{
	writel_relaxed(WDT_CTRL|TWD_TIMER_CONTROL_ENABLE, VA_LOCALTIMER + TWD_WDOG_CONTROL);
}
static void cpu_wdt_stop(void)
{
	writel_relaxed(TWD_WDOG_MAGIC1, VA_LOCALTIMER + TWD_WDOG_DISABLE);
	writel_relaxed(TWD_WDOG_MAGIC2, VA_LOCALTIMER + TWD_WDOG_DISABLE);
	writel_relaxed(0, VA_LOCALTIMER + TWD_WDOG_CONTROL);
}
static void cpu_wdt_restart(void)
{
	writel_relaxed(cpu_wdt_count, VA_LOCALTIMER + TWD_WDOG_LOAD);
}

#if 0
static int cpu_wdt_stat(void)
{
	return readl_relaxed(VA_LOCALTIMER + TWD_WDOG_RESETSTAT) & 1;
}
#endif

static void cpu_wdt_clear(void)
{
	writel_relaxed(1, VA_LOCALTIMER + TWD_WDOG_RESETSTAT);
}

/*-----------------------------------------------------------*/

#define MAX_PRESCALER 256

static void wdt_setup_counts(void)
{
	unsigned long clk, msec;

	/* CPU WDT load value */
	clk = CXD900X0_TWD_CLK;
	cpu_wdt_prescaler = clk / MHZ;
	if (cpu_wdt_prescaler > MAX_PRESCALER)
		cpu_wdt_prescaler = MAX_PRESCALER;
	msec = (clk / cpu_wdt_prescaler) / MSEC_PER_SEC;
	cpu_wdt_count = msec * wdt_timeout;

	/* TIMER WD load value */
	msec = CLOCK_TICK_RATE / MSEC_PER_SEC;
	cxd900x0_wdt_count = (msec + 1) * wdt_timeout; /* to fire CPU WDT before TIMER WD */

	wmb();
}

/*
 * Touches watchdog from sleep.S
 */
void watchdog_touch(void)
{
	if (WDT_OFF == wdt_mode)
		return;

	cpu_wdt_restart();
	cxd900x0_wdt_restart();
}

/*
 * This callback runs from the timer interrupt.
 */
void watchdog_tick(void)
{
	int this_cpu;

	if (WDT_OFF == wdt_mode)
		return;

	this_cpu = smp_processor_id();
	cpu_wdt_restart();
	if (!this_cpu) {
		cxd900x0_wdt_restart();
	}
}

static int __cpuinit
cpu_callback(struct notifier_block *nfb, unsigned long action, void *hcpu)
{
	switch (action) {
	case CPU_STARTING:
	case CPU_STARTING_FROZEN:
		cpu_wdt_stop();
		cpu_wdt_clear();
		cpu_wdt_setup();
		cpu_wdt_start();
		break;
#ifdef CONFIG_HOTPLUG_CPU
	case CPU_DYING:
	case CPU_DYING_FROZEN:
		cpu_wdt_stop();
		break;
#endif /* CONFIG_HOTPLUG_CPU */
	}
	return NOTIFY_OK;
}

static struct notifier_block __cpuinitdata cpu_nfb = {
	.notifier_call = cpu_callback
};

/*
 * Touches watchdog from exception.c
 */
static DEFINE_RAW_SPINLOCK(wdte_lock);
int watchdog_touch_exception(void)
{

	if (WDT_OFF != wdt_mode && !s_except) {
		unsigned long flags;
		char msg[64];
		int cpu;

		raw_spin_lock_irqsave(&wdte_lock, flags);

		/* touch before calculation */
		watchdog_tick();

		/* modify both wdt values */
		cxd900x0_wdt_stop();
		if (wdt_timeout < wdt_timeout_mp_excpt)
			wdt_timeout = wdt_timeout_mp_excpt;
		wdt_setup_counts(); /* update */

		/* reflect immediately on this cpu.
		   not necessary to call cpu_wdt_setup()
		   since CortexA5 specification says that
		   writing TWD_WDOG_LOAD is identical with
		   writing TWD_WDOG_COUNTER */
		cpu_wdt_restart();
		/* alternatively wdt IP requires setup */
		cxd900x0_wdt_setup();
		cxd900x0_wdt_start();
		/* log some info */
		printk(KERN_ERR "watchdog: new_tick=%d\n", wdt_timeout);
		cpu = smp_processor_id();
		snprintf(msg, sizeof(msg)-1, "watchdog extended on cpu%d", cpu);
		boot_time_add(msg);
		/* we'd like to prevent making it chain */
		s_except = 1;

		raw_spin_unlock_irqrestore(&wdte_lock, flags);
	}

	return WDT_OFF != wdt_mode;
}


#define WDOGINTR_NAME "rstreq"
static int wdog_info;

static irqreturn_t wdt_intr(int irq, void *dev_id)
{
	unsigned long stat;

	stat = readl_relaxed(VA_MISC_RSTREQ_STAT);
	printk(KERN_ERR "WDT:RSTREQ_STAT=0x%lx\n", stat);
	if (stat & (MISC_RSTREQ_ST_CPU3|MISC_RSTREQ_ST_CPU2|MISC_RSTREQ_ST_CPU1|MISC_RSTREQ_ST_CPU0)){
		/* clear CPU WDT events */
		writel_relaxed(MISC_RSTREQ_CPUCLR, VA_MISC_RSTREQ_SET);
		writel_relaxed(MISC_RSTREQ_CPUCLR, VA_MISC_RSTREQ_CLR);
		pm_machine_reset(0);
	}
	if (likely(!s_except))		/* to prevent oops chain */
		BUG();

	return IRQ_HANDLED;
}

static void wdt_setup(void)
{
	wdt_setup_counts();
	cxd900x0_wdt_stop();
	cxd900x0_wdt_setup();

	/* RSTREQ intr */
	if (request_irq(IRQ_WDOG, wdt_intr, IRQF_DISABLED, WDOGINTR_NAME,
			&wdog_info)) {
		printk(KERN_ERR "WDT:ERROR:request_irq %d failed.\n", IRQ_WDOG);
	}
}

static void wdt_show(void)
{
#if 0
	unsigned int cpu;

	/* CXD900X0 does not support this function. */
	printk(KERN_ERR "WDT: checking previous status.\n");
	for (cpu = 0; cpu < NR_CPUS; cpu++) {
		if (cpu_wdt_stat(cpu)) {
			printk(KERN_ERR "WDT: CPU#%d occured.\n", cpu);
			/* clear RESET status */
			cpu_wdt_clear(cpu);
		}
	}
	printk(KERN_ERR "WDT: checking done.\n");
#endif
}

/* pre_smp_initcalls */
static int __init wdt_init(void)
{
	if (WDT_OFF == wdt_mode) {
		return 0;
	}

	wdt_show();
	wdt_setup();

	cpu_callback(&cpu_nfb, CPU_STARTING, (void *)0);
	cxd900x0_wdt_start();

	register_cpu_notifier(&cpu_nfb);

	return 0;
}
early_initcall(wdt_init);

void cxd900x0_wdt_resume(void)
{
	if (WDT_OFF == wdt_mode) {
		return ;
	}
	cxd900x0_wdt_stop();
	cxd900x0_wdt_setup();
	cpu_callback(&cpu_nfb, CPU_STARTING, (void *)0);
	cxd900x0_wdt_start();
}

/* ----------------WDT Driver-------------------- */
#define LOCK_ENABLE     1
#define LOCK_DISABLE    0
#define STOP_WDT        0
#define START_WDT       1
#define START_DEBUG     2
#define WDT_IOC_MAGIC   'w'
#define WDT_SET_START   _IOW(WDT_IOC_MAGIC, START_WDT, unsigned int)
#define WDT_SET_STOP    _IO(WDT_IOC_MAGIC, STOP_WDT)
#define WDT_SET_DEBUG   _IOW(WDT_IOC_MAGIC, START_DEBUG, unsigned int)
#define WDT_NAME        "wdt"
#define EMGINTR_NAME    "EMERGENCY"

static DEFINE_SPINLOCK(wdt_lock);
static unsigned char wdt_lock_status = LOCK_DISABLE;

static int wdt_proc_open(struct inode *node, struct file *fp)
{
	if (!spin_trylock(&wdt_lock)) {
		printk(KERN_ERR "[WDT][OPEN1] WDT is Busy.\n");
		return -EBUSY;
	}

	if (wdt_lock_status & LOCK_ENABLE) {
		spin_unlock(&wdt_lock);
		printk(KERN_ERR "[WDT][OPEN2] WDT is Busy.\n");
		return -EBUSY;
	}

	wdt_lock_status = LOCK_ENABLE;
	spin_unlock(&wdt_lock);

	return 0;
}

static int wdt_proc_release(struct inode *node, struct file *fp)
{
	wdt_lock_status = LOCK_DISABLE;
	smp_wmb();

	return 0;
}

static long wdt_proc_ioctl(struct file *fp, unsigned int cmd, unsigned long arg)
{
	long ret = 0;
	printk(KERN_INFO "[INFO][wdt_proc_ioctl] ioctl: cmd=%08X, arg=%u\n", cmd, (unsigned int)arg);

	switch(cmd) {
	case WDT_SET_STOP:
		/* WDT Reset Request disable. */
		writel_relaxed(MISC_RSTREQ_EN, VA_MISC_RSTREQ_CLR);
		cxd900x0_wdt_stop();
		break;
	case WDT_SET_START:
		/* WDT Reset Request enable. */
		cxd900x0_wdt_count = (unsigned int)arg;
		writel_relaxed(MISC_RSTREQ_EN, VA_MISC_RSTREQ_SET);
		writel_relaxed(cxd900x0_wdt_count, VA_WDT_TIMER+CXD90014_TIMERCMP);
		cxd900x0_wdt_start();
		break;
	case WDT_SET_DEBUG:
		cxd900x0_wdt_count = (unsigned int)arg;
		writel_relaxed(MISC_RSTREQ_EN, VA_MISC_RSTREQ_CLR);
		writel_relaxed(cxd900x0_wdt_count, VA_WDT_TIMER+CXD90014_TIMERCMP);
		cxd900x0_wdt_start();
		break;
	default:
		printk(KERN_ERR "[ERR][wdt_proc_ioctl] cmd error. cmd=%08X\n", cmd);
		ret = -1;
		break;
	}

	return ret;
}

static irqreturn_t emg_intr(int irq, void *dev_id)
{
	unsigned long rstreq_stat;

	rstreq_stat = readl_relaxed(VA_MISC_RSTREQ_STAT);
	if (rstreq_stat & MISC_RSTREQ_ST_TIM) {
		printk(KERN_ERR "START Emegency!\n");
		boot_time_add("Emergency interrupt!!");
		cxd900x0_wdt_stop();
	}
	if (rstreq_stat & (MISC_RSTREQ_ST_CPU3|MISC_RSTREQ_ST_CPU2|MISC_RSTREQ_ST_CPU1|MISC_RSTREQ_ST_CPU0)){
		/* clear WDT events */
		writel_relaxed(MISC_RSTREQ_CPUCLR, VA_MISC_RSTREQ_SET);
		writel_relaxed(MISC_RSTREQ_CPUCLR, VA_MISC_RSTREQ_CLR);
	}

	rstreq_stat = readl_relaxed(VA_MISC_RSTREQ);
	if (rstreq_stat == 0) {
		errind_start(ERRIND_EMG);
	}
	if (rstreq_stat & MISC_RSTREQ_ASSERT) {
		/* asserted by Exception Monitor or Reboot
		 *  Do not clear RMAN bit.
		 */
		disable_irq_nosync(IRQ_WDOG);
	}

	return IRQ_HANDLED;
}

static struct file_operations wdt_operations = {
	.open               = wdt_proc_open,
	.unlocked_ioctl     = wdt_proc_ioctl,
	.release            = wdt_proc_release,
};

static UDIF_ERR wdt_proc_init( UDIF_VP data )
{
	struct proc_dir_entry *wdt_entry;

	wdt_entry = create_proc_entry(WDT_NAME, S_IRUSR | S_IWUSR, NULL);
	if (wdt_entry) {
		wdt_entry->proc_fops = &wdt_operations;
	} else {
		printk(KERN_ERR "ERROR:%s:create_proc_entry\n", __func__);
	}

	if (WDT_OFF == wdt_mode) {
		if (request_irq(IRQ_WDOG, emg_intr, IRQF_DISABLED, EMGINTR_NAME, &wdog_info)) {
			printk(KERN_ERR "WDT:ERROR:request_irq %d failed.\n", IRQ_WDOG);
		}
	}
	return UDIF_ERR_OK;
}

static UDIF_ERR wdt_proc_exit( UDIF_VP data )
{
	remove_proc_entry(WDT_NAME, NULL);

	return UDIF_ERR_OK;
}

#if 0 /* This is achieved by sleep.S */
static UDIF_ERR wdt_proc_suspend( const UDIF_DEVICE *dev, UDIF_CH ch, UDIF_VP data )
{
	printk(KERN_ERR "[wdt] WDT Suspend.\n");

	boot_time_add("WDT Reset");
	writel_relaxed(0, VA_MISC_RSTREQ);
	cxd900x0_wdt_stop();

	return UDIF_ERR_OK;
}
#endif

static UDIF_DRIVER_OPS wdt_ops = {
	.init = wdt_proc_init,
	.exit = wdt_proc_exit,
#if 0
	.suspend = wdt_proc_suspend,
#endif
};

UDIF_MODULE( wdt, "wdt", "2.0", wdt_ops, NULL, NULL, NULL );
