/*
 * fs/core_lz77.c
 *
 * Copyright 2017 Sony Corporation
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  version 2 of the  License.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
 */

#include <linux/module.h>
#include <linux/coredump.h>
#include <linux/lz77.h>

/* SEEK record definition (for coredump only)
     "0E dd cc bb aa" represents seek 0xaabbccdd bytes */
#define LZSEEK (0x0E)
#define LZSEEK_REC_SIZE  5

#define LZ77_RECORDSIZE 4096
static unsigned char lz77dst[LZ77_RECORDSIZE + LZ77_RECORDSIZE/2 + 4];
static unsigned char lz77work[LZ77_WORKSPACE_SIZE(LZ77_RECORDSIZE)];

static DEFINE_MUTEX(core_lz77_mutex);

static int lz77_seek(struct file *file, loff_t off)
{
	u32 skip;
	int ret;
	unsigned char dst[LZSEEK_REC_SIZE];

	if (off < 0)
		return -1;
	if (!off)
		return 0;

	skip = (u32)off;

	dst[0] = LZSEEK;
	dst[1] = skip;
	dst[2] = skip >> 8;
	dst[3] = skip >> 16;
	dst[4] = skip >> 24;

	ret = file->f_op->write(file, dst, LZSEEK_REC_SIZE, &file->f_pos);
	if (LZSEEK_REC_SIZE != ret) {
		printk(KERN_ERR "%s:ERROR:skip=%u,ret=%d\n", __func__,
		       skip, ret);
		return -1;
	}

	return 0;
}

static int lz77_write(struct file *file, const void *addr, int nr)
{
	int ret = 0;
	unsigned char *p = (unsigned char *)addr;

	mutex_lock(&core_lz77_mutex);
	while (nr > 0) {
		int len, olen;

		len = nr;
		if (len > LZ77_RECORDSIZE)
			len = LZ77_RECORDSIZE;

		olen = lz77_deflate_fast(p, len,
					 lz77dst, LZ77_RECORDSIZE+4,
					 (void *)lz77work, sizeof lz77work);
		if (olen <= 0) {
			printk(KERN_ERR "%s:ERROR:olen=%d\n", __func__, olen);
			ret = -1;
			break;
		}
		ret = file->f_op->write(file, lz77dst, olen, &file->f_pos);
		if (ret != olen) {
			printk(KERN_ERR "%s:ERROR:ret=%d\n", __func__, ret);
			ret = -1;
			break;
		}

		p += len;
		nr -= len;
	}
	mutex_unlock(&core_lz77_mutex);
	return ret;
}

struct corewriter core_lz77 = {
	.name	= "lz77",
	.write 	= lz77_write,
	.seek	= lz77_seek,
};
